interface EmailNotification {
  to: string;
  subject: string;
  template: string;
  data: any;
}

interface ProjectAwardedEmailData {
  freelancerName: string;
  projectTitle: string;
  clientName: string;
  bidAmount: number;
  projectUrl: string;
}

interface ProjectCancelledEmailData {
  freelancerName: string;
  projectTitle: string;
  clientName: string;
  reason: string;
  bidAmount?: number;
}

interface BidReceivedEmailData {
  clientName: string;
  projectTitle: string;
  freelancerName: string;
  bidAmount: number;
  projectUrl: string;
}

// Email templates
const emailTemplates = {
  PROJECT_AWARDED: (data: ProjectAwardedEmailData) => ({
    subject: `🎉 Congratulations! Your bid was selected for "${data.projectTitle}"`,
    html: `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
        <h2 style="color: #4CAF50;">Congratulations ${data.freelancerName}!</h2>
        <p>Great news! Your bid has been selected for the project:</p>
        
        <div style="background: #f5f5f5; padding: 20px; margin: 20px 0; border-radius: 8px;">
          <h3 style="margin: 0 0 10px 0; color: #333;">${data.projectTitle}</h3>
          <p style="margin: 5px 0;"><strong>Client:</strong> ${data.clientName}</p>
          <p style="margin: 5px 0;"><strong>Your Winning Bid:</strong> €${data.bidAmount}</p>
        </div>
        
        <p>You can now start working on this project. Please:</p>
        <ul>
          <li>Review the project requirements carefully</li>
          <li>Communicate with the client through the project chat</li>
          <li>Deliver high-quality work on time</li>
        </ul>
        
        <a href="${data.projectUrl}" style="display: inline-block; background: #4CAF50; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; margin: 20px 0;">
          View Project
        </a>
        
        <p style="color: #666; font-size: 14px; margin-top: 30px;">
          Best of luck with your project!<br>
          The Smile24 Team
        </p>
      </div>
    `
  }),

  PROJECT_CANCELLED: (data: ProjectCancelledEmailData) => ({
    subject: `Project "${data.projectTitle}" has been cancelled`,
    html: `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
        <h2 style="color: #FF6B35;">Project Cancelled</h2>
        <p>Hi ${data.freelancerName},</p>
        <p>We're sorry to inform you that the project you bid on has been cancelled:</p>
        
        <div style="background: #f5f5f5; padding: 20px; margin: 20px 0; border-radius: 8px;">
          <h3 style="margin: 0 0 10px 0; color: #333;">${data.projectTitle}</h3>
          <p style="margin: 5px 0;"><strong>Client:</strong> ${data.clientName}</p>
          <p style="margin: 5px 0;"><strong>Reason:</strong> ${data.reason}</p>
          ${data.bidAmount ? `<p style="margin: 5px 0;"><strong>Your Bid:</strong> €${data.bidAmount}</p>` : ''}
        </div>
        
        <p>Don't worry! There are many other great projects available. Keep applying to projects that match your skills.</p>
        
        <a href="/freelancer/projects" style="display: inline-block; background: #4CAF50; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; margin: 20px 0;">
          Find New Projects
        </a>
        
        <p style="color: #666; font-size: 14px; margin-top: 30px;">
          Thank you for your understanding.<br>
          The Smile24 Team
        </p>
      </div>
    `
  }),

  BID_RECEIVED: (data: BidReceivedEmailData) => ({
    subject: `New bid received for "${data.projectTitle}"`,
    html: `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
        <h2 style="color: #4CAF50;">New Bid Received!</h2>
        <p>Hi ${data.clientName},</p>
        <p>You've received a new bid for your project:</p>
        
        <div style="background: #f5f5f5; padding: 20px; margin: 20px 0; border-radius: 8px;">
          <h3 style="margin: 0 0 10px 0; color: #333;">${data.projectTitle}</h3>
          <p style="margin: 5px 0;"><strong>Freelancer:</strong> ${data.freelancerName}</p>
          <p style="margin: 5px 0;"><strong>Bid Amount:</strong> €${data.bidAmount}</p>
        </div>
        
        <p>Review the bid details and freelancer profile to make an informed decision.</p>
        
        <a href="${data.projectUrl}" style="display: inline-block; background: #4CAF50; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; margin: 20px 0;">
          Review Bid
        </a>
        
        <p style="color: #666; font-size: 14px; margin-top: 30px;">
          Happy hiring!<br>
          The Smile24 Team
        </p>
      </div>
    `
  })
};

export class EmailService {
  static async sendProjectAwardedEmail(
    freelancerEmail: string, 
    data: ProjectAwardedEmailData
  ): Promise<boolean> {
    const template = emailTemplates.PROJECT_AWARDED(data);
    return this.sendEmail({
      to: freelancerEmail,
      subject: template.subject,
      template: template.html,
      data
    });
  }

  static async sendProjectCancelledEmail(
    freelancerEmail: string,
    data: ProjectCancelledEmailData
  ): Promise<boolean> {
    const template = emailTemplates.PROJECT_CANCELLED(data);
    return this.sendEmail({
      to: freelancerEmail,
      subject: template.subject,
      template: template.html,
      data
    });
  }

  static async sendBidReceivedEmail(
    clientEmail: string,
    data: BidReceivedEmailData
  ): Promise<boolean> {
    const template = emailTemplates.BID_RECEIVED(data);
    return this.sendEmail({
      to: clientEmail,
      subject: template.subject,
      template: template.html,
      data
    });
  }

  private static async sendEmail(notification: EmailNotification): Promise<boolean> {
    try {
      // In production, integrate with your email service (SendGrid, AWS SES, etc.)
      console.log('📧 Email would be sent:', {
        to: notification.to,
        subject: notification.subject,
        html: notification.template
      });

      // Mock successful email sending
      return true;
    } catch (error) {
      console.error('Failed to send email:', error);
      return false;
    }
  }
}

export type { ProjectAwardedEmailData, ProjectCancelledEmailData, BidReceivedEmailData };