"use client";

import { useState, useEffect, useRef, useCallback } from "react";
import { useSession } from "next-auth/react";
import MessageInput from "./MessageInput";
import BidActionHeader from "./BidActionHeader";
import { useWebSocketContext } from "@/context/WebSocketContext";
import AttachmentModal from "./AttachmentModal";

interface Profile {
  firstName?: string;
  lastName?: string;
  profileImage?: string;
}

interface User {
  id: string;
  email: string;
  profile?: Profile;
}

interface Message {
  id: string;
  content: string;
  attachments?: string;
  senderId: string;
  receiverId: string;
  sender: User;
  receiver: User;
  isRead: boolean;
  readAt?: string;
  createdAt: string;
  project?: {
    id: string;
    title: string;
  };
  isOptimistic?: boolean; // Flag for optimistic messages
}

interface MessageThreadProps {
  contact?: User | null;
  conversationId: string;
  projectId?: string;
  projectStatus?: string;
  handleBackClick: () => void;
  showSidebar: boolean;
}

export default function MessageThread({
  contact,
  conversationId,
  projectId,
  projectStatus,
  handleBackClick,
  showSidebar,
}: MessageThreadProps) {
  // Check if project is archived (completed or cancelled)
  const isProjectArchived =
    projectStatus === "completed" || projectStatus === "cancelled";
  console.log("🔍 [MessageThread] Project status:", projectStatus);
  const { data: session } = useSession();
  const [messages, setMessages] = useState<Message[]>([]);
  const [loading, setLoading] = useState(false);
  const [page, setPage] = useState(1);
  const [hasMore, setHasMore] = useState(false);
  const [isTyping, setIsTyping] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const messagesContainerRef = useRef<HTMLDivElement>(null);
  const typingTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  const [loadingMore, setLoadingMore] = useState(false);

  const scrollPositionRef = useRef<number>(0);
  const isInitialLoadRef = useRef(true);

  const [messageIds, setMessageIds] = useState<Set<string>>(new Set());


  const shouldScrollToBottomRef = useRef(false);
  const prevMessagesLengthRef = useRef(0);


  const addMessageToState = useCallback((message: Message) => {
    setMessages(prev => {
      // Check if message already exists in current messages
      const existsInCurrent = prev.some(m => m.id === message.id);
      if (existsInCurrent) {
        return prev;
      }

      // Add to IDs set
      setMessageIds(prevIds => new Set(prevIds).add(message.id));

      // Replace optimistic message if timestamps are close & same sender/receiver
      const currentUserId = (session?.user as any)?.id;
      if (message.senderId === currentUserId) {
        const optimisticIndex = prev.findIndex(
          (m) =>
            m.isOptimistic &&
            m.senderId === message.senderId &&
            m.receiverId === message.receiverId &&
            Math.abs(new Date(m.createdAt).getTime() - new Date(message.createdAt).getTime()) < 5000
        );

        if (optimisticIndex !== -1) {
          const newArr = [...prev];
          newArr[optimisticIndex] = { ...message, isOptimistic: false };
          return newArr;
        }
      }

      // Append at end (newest at the bottom)
      return [...prev, message];
    });
  }, [session?.user]);

  const {
    isConnected,
    sendMessage: sendWebSocketMessage,
    joinRoom,
    leaveRoom,
    lastMessage,
  } = useWebSocketContext();

  useEffect(() => {
    if (lastMessage) {
      console.log(
        "🎯 [MessageThread] WebSocket message received in useEffect:",
        lastMessage
      );
      try {
        const message =
          typeof lastMessage === "string"
            ? JSON.parse(lastMessage)
            : lastMessage;
        console.log("🔍 [MessageThread] Parsed message:", message);

        switch (message.type) {
          case "new_message":
            console.log("📨 [MessageThread] Handling new_message case");
            handleNewMessage(message);
            break;

          case "typing_start":
            handleTypingStart(message);
            break;

          case "typing_stop":
            handleTypingStop(message);
            break;

          case "message_read":
            handleMessageRead(message);
            break;

          case "user_online":
          case "user_offline":
            // Could update user presence indicator
            break;

            case "read_receipt":
            // Handle read receipt
            handleReadRecipt(message);
            break;

          case "message_sent":
            // Message sent confirmation
            break;

          case "error":
            console.warn("WebSocket error:", message.error);
            break;

          default:
            console.log("Unknown WebSocket message type:", message.type);
        }
      } catch (error) {
        console.error(
          "❌ [MessageThread] Error processing WebSocket message:",
          error
        );
      }
    }
  }, [lastMessage]);

  const handleReadRecipt = (message: any) => {
    console.log("🔄 [MessageThread] Handling read_receipt:", message);
    // const messageData = message.message;

    const { conversationId: convId, messageId, timestamp, senderId, receiverId } = message.message;

    if (convId === conversationId) {
      // active conversation on both sides, make the messages as read in real time
      setMessages((prev) => {
        return prev.map((m) => {

          return { ...m, isRead: true };

        });
      });
    }
  }

  // Handle new message WebSocket event
  const handleNewMessage = (message: any) => {

    console.log("🔄 [MessageThread] Handling new_message:", message);
    if (!message.message) {
      console.log("❌ [MessageThread] No message data in new_message event");
      return;
    }

    // Use session user email consistently (matches WebSocket authentication)
    const currentUserEmail = session?.user?.email;
    const currentUserId = (session?.user as any)?.id;
    const msgSenderId = message.message.senderId;
    const msgReceiverId = message.message.receiverId;
    const conversation = message.message.conversationId;

    console.log("🔍 [MessageThread] User IDs:", {
      currentUserEmail,
      currentUserId,
      msgSenderId,
      msgReceiverId,
      contactId: contact?.id,
      contactEmail: contact?.email,
    });

    // Check if this message is relevant to the current conversation
    // Use ID for comparison since that's what the database uses
    const isRelevant =
      conversation === conversationId &&
      contact?.id &&
      ((msgSenderId === contact.id && msgReceiverId === currentUserId) ||
        (msgSenderId === currentUserId && msgReceiverId === contact.id));

    console.log('isRelevant: ', isRelevant);

    console.log('first: ', isRelevant, ' contact: ', contact, ' currentUserId: ', currentUserId, ' msgSenderId: ', msgSenderId, ' msgReceiverId: ', msgReceiverId);

    console.log("🔍 [MessageThread] Message relevance check:", {
      isRelevant,
      isFromContact: msgSenderId === contact?.id,
      isToCurrentUser: msgReceiverId === currentUserId,
      isFromCurrentUser: msgSenderId === currentUserId,
      isToContact: msgReceiverId === contact?.id,
    });

    // Process all relevant messages (including our own for real-time updates)
    if (isRelevant) {
      console.log("✅ [MessageThread] Adding new message to conversation");
      addMessageToState(message.message);

      // setMessages((prev) => {
      //   // Check for existing message by ID
      //   const existsById = prev.some((m) => m.id === message.message.id);

      //   // For messages from current user, check if we have an optimistic version
      //   if (msgSenderId === currentUserId) {
      //     const hasOptimistic = prev.some(
      //       (m) =>
      //         m.isOptimistic &&
      //         m.senderId === msgSenderId &&
      //         m.receiverId === msgReceiverId &&
      //         Math.abs(
      //           new Date(m.createdAt).getTime() -
      //           new Date(message.message.createdAt).getTime()
      //         ) < 5000
      //     );

      //     if (hasOptimistic && !existsById) {
      //       // Replace optimistic message with real one
      //       console.log(
      //         "🔄 [MessageThread] Replacing optimistic message with real WebSocket message"
      //       );
      //       return prev.map((m) =>
      //         m.isOptimistic &&
      //           m.senderId === msgSenderId &&
      //           m.receiverId === msgReceiverId &&
      //           Math.abs(
      //             new Date(m.createdAt).getTime() -
      //             new Date(message.message.createdAt).getTime()
      //           ) < 5000
      //           ? { ...message.message, isOptimistic: false }
      //           : m
      //       );
      //     } else if (existsById) {
      //       // Real message already exists, skip
      //       console.log(
      //         "⚠️ [MessageThread] Real message already exists, skipping"
      //       );
      //       return prev;
      //     }
      //   }

      //   // For new messages (not duplicates)
      //   if (!existsById) {
      //     const newMessages = [...prev, message.message];

      //     // Auto-scroll to bottom for new messages
      //     setTimeout(() => scrollToBottom(true), 100);

      //     // Mark message as read if tab is active and message is from other user
      //     if (!document.hidden && msgSenderId !== currentUserId) {
      //       setTimeout(() => markMessageAsRead(message.message.id), 500);
      //     }

      //     return newMessages;
      //   }

      //   console.log("⚠️ [MessageThread] Message already exists, skipping");
      //   return prev;
      // });

      const container = messagesContainerRef.current;
      const isNearBottom = () => {
        if (!container) return true;
        const threshold = 150; // px from bottom
        const distanceFromBottom = container.scrollHeight - container.scrollTop - container.clientHeight;
        return distanceFromBottom <= threshold;
      };


      setTimeout(() => markMessageAsRead(message.message.id), 0);
      if (msgSenderId === currentUserId) {
        // It's our message — scroll to bottom to show confirmation.
        shouldScrollToBottomRef.current = true;
      } else if (isNearBottom()) {
        // Friend sent message and user is at/near bottom — auto-scroll
        shouldScrollToBottomRef.current = true;
        // Mark message as read shortly (if tab not hidden)
        // if (!document.hidden) {
        //   setTimeout(() => markMessageAsRead(message.message.id), 0);
        // }
      } else {
        // Not near bottom: do not auto-scroll (user is reading older messages)
        shouldScrollToBottomRef.current = false;
      }

      // Show browser notification for new messages when tab is not active (only for messages from others)
      if (document.hidden && msgSenderId !== currentUserId) {
        showNotification(message.message);
      }
    } else {
      console.log(
        "❌ [MessageThread] Message not relevant or from current user, ignoring"
      );
    }
  };

  // Handle typing start WebSocket event
  const handleTypingStart = (message: any) => {
    if (message.userId === contact?.id || message.userId === contact?.email) {
      setIsTyping(true);
      // Clear any existing timeout
      if (typingTimeoutRef.current) {
        clearTimeout(typingTimeoutRef.current);
      }
      // Auto-stop typing after 5 seconds if no typing_stop received
      typingTimeoutRef.current = setTimeout(() => {
        setIsTyping(false);
      }, 5000);
    }
  };

  // Handle typing stop WebSocket event
  const handleTypingStop = (message: any) => {
    if (message.userId === contact?.id || message.userId === contact?.email) {
      setIsTyping(false);
      if (typingTimeoutRef.current) {
        clearTimeout(typingTimeoutRef.current);
        typingTimeoutRef.current = null;
      }
    }
  };

  // Handle message read WebSocket event
  const handleMessageRead = (message: any) => {
    if (message.messageIds && Array.isArray(message.messageIds)) {
      setMessages((prev) =>
        prev.map((msg) =>
          message.messageIds.includes(msg.id)
            ? {
              ...msg,
              isRead: true,
              readAt: message.readAt || new Date().toISOString(),
            }
            : msg
        )
      );
    }
  };

  useEffect(() => {
    console.log("🔄 [MessageThread] Main effect triggered:", {
      hasSession: !!session?.user?.email,
      contactId: contact?.id,
      isConnected,
      sessionUserId: (session?.user as any)?.id,
    });

    if (session?.user?.email && contact?.id) {
      console.log(
        "💼 [MessageThread] Initializing conversation with:",
        contact.id
      );
      setMessages([]);
      // reset the set
      setMessageIds(new Set());
      setPage(1);

      isInitialLoadRef.current = true;
      fetchMessages(1, true);
      markConversationAsRead();

      // Join conversation room for real-time updates using consistent room naming
      if (isConnected && session?.user) {
        const currentUserId = (session.user as any)?.id;
        // Create a consistent room ID regardless of who initiates the conversation
        const roomId = `conversation_${currentUserId < contact.id ? currentUserId : contact.id
          }_${currentUserId < contact.id ? contact.id : currentUserId}`;
        console.log(
          `🚪 [MessageThread] Joining room: ${roomId} (isConnected: ${isConnected})`
        );
        joinRoom(roomId);
      } else {
        console.log(
          "⚠️ [MessageThread] Not connected to WebSocket yet, cannot join room"
        );
      }
    }

    return () => {
      // Leave conversation room on cleanup
      if (isConnected && contact?.id && session?.user) {
        const currentUserId = (session.user as any)?.id;
        const roomId = `conversation_${currentUserId < contact.id ? currentUserId : contact.id
          }_${currentUserId < contact.id ? contact.id : currentUserId}`;
        console.log(`🚪 [MessageThread] Leaving room: ${roomId}`);
        leaveRoom(roomId);
      }
    };
    // }, [contact?.id, projectId, isConnected, joinRoom, leaveRoom, session]);
  }, [contact?.id, projectId, isConnected, joinRoom, leaveRoom]);

  useEffect(() => {
    const prevLen = prevMessagesLengthRef.current;
    const currLen = messages.length;

    // Save new length for next change
    prevMessagesLengthRef.current = currLen;

    // If initial load, scroll to bottom once
    if (isInitialLoadRef.current && currLen > 0) {
      isInitialLoadRef.current = false;
      scrollToBottom(false);
      return;
    }

    if (shouldScrollToBottomRef.current) {
      scrollToBottom(true);
      shouldScrollToBottomRef.current = false;
      return;
    }

    // return () => clearTimeout(timer);
  }, [messages]);


  useEffect(() => {
    const messagesContainer = messagesContainerRef.current;
    if (!messagesContainer) return;

    const handleScroll = () => {
      // Only trigger when scrolled to very top
      if (messagesContainer.scrollTop <= 0 && hasMore && !loadingMore && !loading) {
        loadMoreMessages();
      }
    };

    messagesContainer.addEventListener("scroll", handleScroll);
    return () => messagesContainer.removeEventListener("scroll", handleScroll);
  }, [hasMore, loadingMore, loading]);

  // // Additional effect to scroll when component first loads
  // useEffect(() => {
  //   if (messages.length > 0) {
  //     const timer = setTimeout(() => {
  //       scrollToBottom();
  //     }, 200);

  //     return () => clearTimeout(timer);
  //   }
  // }, [messages.length]);

  // Initialize notification permissions
  useEffect(() => {
    requestNotificationPermission();
  }, []);

  const fetchMessages = async (
    pageNum: number = page,
    reset: boolean = false
  ) => {
    try {
      if (reset) {

        setLoading(true);
      } else {
        setLoadingMore(true);
      }

      if (!contact?.id) return;

      const params = new URLSearchParams({
        contactId: contact.id,
        page: pageNum.toString(),
        limit: "10",
        conversationId,
      });

      if (projectId) params.append("projectId", projectId);

      const response = await fetch(`/api/messages/conversations/${contact.id}?${params}`);
      if (!response.ok) {
        console.error("Failed to fetch messages:", response.statusText);
        return;
      }

      const data = await response.json();
      const fetchedMessages: Message[] = data.data.messages || [];

      if (reset) {
        // Fresh load (initial or explicit refresh) — replace messages
        setMessages(fetchedMessages);
        setMessageIds(new Set(fetchedMessages.map(msg => msg.id)));
        setHasMore(data.data.pagination.hasNext);
      } else {
        // Loading older messages (prepend)
        const container = messagesContainerRef.current;
        if (container) {
          // Save current scroll distance from bottom so we can restore after prepending
          scrollPositionRef.current = container.scrollHeight - container.scrollTop;
        }

        const uniqueMessages = fetchedMessages.filter(msg => !messageIds.has(msg.id));

        if (uniqueMessages.length > 0) {
          // Add new IDs to the set
          setMessageIds(prev => {
            const newSet = new Set(prev);
            uniqueMessages.forEach(msg => newSet.add(msg.id));
            return newSet;
          });

          setMessages((prev) => {
            return [...uniqueMessages, ...prev];
          });
        }

        // setMessages((prev) => {
        //   return [...fetchedMessages, ...prev];
        // });

        // Wait for DOM update, then restore scroll so user's viewport stays where it was
        setTimeout(() => {
          const c = messagesContainerRef.current;
          if (c) {
            c.scrollTop = c.scrollHeight - scrollPositionRef.current;
          }
        }, 0);

        setHasMore(data.data.pagination.hasNext);
      }
    } catch (err) {
      console.error("Error fetching messages:", err);
    } finally {
      await new Promise((resolve) => setTimeout(resolve, 10000));
      setLoading(false);
      setLoadingMore(false);
    }
  };

  const loadMoreMessages = useCallback(() => {
    if (!loadingMore && !loading && hasMore) {
      const nextPage = page + 1;
      setPage(nextPage);
      fetchMessages(nextPage);
    }
  }, [loadingMore, loading, hasMore, page]);

  const markConversationAsRead = async () => {
    try {
      await fetch(`/api/messages/conversations/${contact?.id}`, {
        method: "PUT",
        body: JSON.stringify({ projectId, conversationId, receiverId: contact?.id}),
      });
    } catch (error) {
      console.error("Error marking conversation as read:", error);
    }
  };

  const markMessageAsRead = async (messageId: string) => {
    console.log('Updating the Message read status: ', messageId);
    try {
      await fetch(`/api/messages/${messageId}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ isRead: true, conversationId, projectId, receiverId: contact?.id }),
      });

      // Send WebSocket notification about read status
      if (isConnected) {
        sendWebSocketMessage({
          type: "message_read",
          messageId,
          receiverId: contact?.email || contact?.id || "",
        });
      }
    } catch (error) {
      console.error("Error marking message as read:", error);
    }
  };

  const showNotification = (message: Message) => {
    if ("Notification" in window && Notification.permission === "granted") {
      const senderName = getDisplayName(message.sender);
      new Notification(`New message from ${senderName}`, {
        body:
          message.content.slice(0, 100) +
          (message.content.length > 100 ? "..." : ""),
        icon: getProfileImage(message.sender),
        tag: `message-${message.id}`,
      });
    }
  };

  const requestNotificationPermission = () => {
    if ("Notification" in window && Notification.permission === "default") {
      Notification.requestPermission();
    }
  };


  const sendMessage = async (content: string, attachments?: any[]) => {
    const currentUserId = (session?.user as any)?.id;
    if (!contact) throw new Error("No contact selected");

    const optimisticMessage: Message = {
      id: `temp_${Date.now()}`,
      content: content.trim(),
      attachments: attachments ? JSON.stringify(attachments) : undefined,
      senderId: currentUserId,
      receiverId: contact.id,
      sender: {
        id: currentUserId,
        email: session?.user?.email || "",
        profile: {
          firstName: session?.user?.name?.split(" ")[0] || "",
          lastName: session?.user?.name?.split(" ")[1] || "",
          profileImage: session?.user?.image || "/assets/image/harveyspecter.png",
        },
      },
      receiver: contact,
      isRead: false,
      readAt: undefined,
      createdAt: new Date().toISOString(),
      project: projectId ? { id: projectId, title: "Project" } : undefined,
      isOptimistic: true,
    };

    try {
      // Append optimistic message
      setMessages((prev) => [...prev, optimisticMessage]);
      setMessageIds(prevIds => new Set(prevIds).add(optimisticMessage.id));

      // On next messages render we'll scroll to bottom because we set flag
      shouldScrollToBottomRef.current = true;

      // Send request to server
      const response = await fetch("/api/messages", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          receiverId: contact.id,
          content,
          attachments,
          projectId,
          conversationId,
        }),
      });

      if (!response.ok) {
        setMessages((prev) => prev.filter((m) => m.id !== optimisticMessage.id));
        const errorData = await response.json();
        throw new Error(errorData.error || "Failed to send message");
      } else {
      }
    } catch (err) {
      // ensure optimistic message removed on error
      setMessages((prev) => prev.filter((m) => m.id !== optimisticMessage.id));
      throw err;
    }
  };

  const handleTyping = (isTyping: boolean) => {
    if (isConnected && contact) {
      sendWebSocketMessage({
        type: isTyping ? "typing_start" : "typing_stop",
        receiverId: contact.email || contact.id,
      });
    }
  };

  const scrollToBottom = (smooth: boolean = false) => {
    const c = messagesContainerRef.current;
    if (!c) return;
    requestAnimationFrame(() => {
      if (smooth) {
        c.scrollTo({ top: c.scrollHeight, behavior: "smooth" });
      } else {
        c.scrollTop = c.scrollHeight;
      }
    });
  };

  const formatTime = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleTimeString("en-US", {
      hour: "2-digit",
      minute: "2-digit",
      hour12: true,
    });
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);

    if (date.toDateString() === today.toDateString()) {
      return "Today";
    } else if (date.toDateString() === yesterday.toDateString()) {
      return "Yesterday";
    } else {
      return date.toLocaleDateString("en-US", {
        month: "short",
        day: "numeric",
        year:
          date.getFullYear() !== today.getFullYear() ? "numeric" : undefined,
      });
    }
  };

  const getDisplayName = (user: User) => {
    if (user.profile?.firstName && user.profile?.lastName) {
      return `${user.profile.firstName} ${user.profile.lastName}`;
    }
    return user.email;
  };

  const getProfileImage = (user: User) => {
    return user.profile?.profileImage || "/assets/image/harveyspecter.png";
  };

  const [selectedAttachment, setSelectedAttachment] = useState<{
    url: string;
    type: string;
    name?: string;
  } | null>(null);

  const renderAttachments = (attachments: string) => {
    if (JSON.parse(attachments).length === 0) return null;

    try {
      const attachmentsArray = JSON.parse(attachments);

      return (
        <>
          <div className="attachments-list">
            {attachmentsArray.map((attachment: any, idx: number) => {
              const fileUrl = process.env.NEXT_PUBLIC_APP_URL + attachment.url;
              const fileName = attachment.name?.toLowerCase() || '';
              const fileType = attachment.type || '';

              const isImage = fileType.startsWith('image/') ||
                /\.(jpg|jpeg|png|gif|bmp|webp|svg)$/i.test(fileName);


              const isVideo = fileType.startsWith('video/') ||
                /\.(mp4|mov|avi|wmv|flv|webm|mkv)$/i.test(fileName);


              const isPDF = fileType === 'application/pdf' || fileName.endsWith('.pdf') || fileType.endsWith("/docs");

              if (isImage) {
                return (
                  <div key={idx} >
                    <img
                      src={fileUrl}
                      alt={attachment.name || 'Image attachment'}
                      onClick={() => {
                        setSelectedAttachment({
                          url: fileUrl,
                          type: fileType,
                          name: attachment.name
                        });
                      }}
                      className="clickable-image"
                    />
                  </div>
                );
              } else if (isVideo) {
                return (
                  <div key={idx} onClick={() => {
                    setSelectedAttachment({
                      url: fileUrl,
                      type: fileType,
                      name: attachment.name
                    });
                  }}>
                    <video
                      controls
                      preload="metadata"
                      className="message-video"
                    >
                      <source src={fileUrl} type={fileType || 'video/mp4'} />
                      Your browser does not support the video tag.
                    </video>
                  </div>
                );
              } else {
                // Default file attachment
                return (
                  <div key={idx}>
                    <a
                      href={fileUrl}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="file-link"
                      download={attachment.name || true}
                    >
                      {attachment.name || 'Download file'}
                    </a>
                  </div>
                );
              }
            })}
          </div>
        </>
      );
    } catch (error) {
      console.error('Error parsing attachments:', error);
      return null;
    }
  };

  const renderMessage = (message: Message, index: number) => {
    const isOwnMessage = message.senderId === (session?.user as any)?.id;
    const isOptimistic = message.isOptimistic;

    return (
      <li
        key={message.id}
        className={`${isOwnMessage ? "replies" : "sent"} ${isOptimistic ? "optimistic" : ""
          }`}
      >
        <img
          src={getProfileImage(message.sender)}
          alt={getDisplayName(message.sender)}
          onError={(e) => {
            e.currentTarget.onerror = null;
            e.currentTarget.src = '/assets/images/user.png';
          }}
        />
        <div>
          <p style={isOptimistic ? { opacity: 0.7, fontStyle: "italic" } : {}}>
            {message.content}
          </p>
          <div className="message-meta-info">
            <span className="message-time">
              {formatTime(message.createdAt)}
            </span>
            {isOwnMessage && (
              <span
                className={`read-receipt ${message.isRead ? "read" : "sent"}`}
              >
                {isOptimistic ? "⏳" : message.isRead ? "✓✓" : "✓"}
              </span>
            )}
          </div>

          {message.attachments && renderAttachments(message.attachments)}
        </div>
      </li>
    );
  };

  if (!session?.user) {
    return null;
  }

  const [isMobile, setIsMobile] = useState<boolean>(false);

  useEffect(() => {
    const checkMobile = () => {
      setIsMobile(window.innerWidth < 768); // 768px is typical breakpoint for mobile
    };

    // Initial check
    checkMobile();

    // Add resize listener
    window.addEventListener('resize', checkMobile);

    return () => {
      window.removeEventListener('resize', checkMobile);
    };
  }, []);

  return (
    <div className={`content chat-box ${showSidebar ? "" : "chat-hidden-box"}`}>
      <div className="contact-profile">
        {contact && (
          <div className="chat-profile-left-header d-flex align-items-center justify-content-between">
            {isMobile && (
              <div className="back-chat-btn cursor-pointer" onClick={() => {
                handleBackClick();
              }}>
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M15 18L9 12L15 6" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"></path>
                </svg>
              </div>
            )}
            <div className="chat-profile-data">
              <img src={getProfileImage(contact)} alt={getDisplayName(contact)} />
              <div className="user-name-chat">
                <p>{getDisplayName(contact)}</p>
              </div>
            </div>
          </div>
        )}

        {/* Show bid action header for clients when there's a project context */}
        {projectId && contact && (
          <BidActionHeader
            contactId={contact.id}
            projectId={projectId}
            onBidAction={(bidId, action) => {
              // Refresh messages after bid action to show system messages
              fetchMessages(1, true);
            }}
          />
        )}
      </div>
      <div className="messages" ref={messagesContainerRef}>
        <ul>
          {hasMore && (
            <li className="loading-center" style={{ textAlign: "center", margin: "20px 0" }}>
              <div
                onClick={loadMoreMessages}
              >
                {loading ? "Loading..." : "Load older messages..."}
              </div>
            </li>
          )}

          {loading && messages.length === 0 ? (
            <li className="loading-center" style={{ textAlign: "center", padding: "40px" }}>
              <div>Loading...</div>
            </li>
          ) : messages.length === 0 ? (
            <li style={{ textAlign: "center", padding: "40px" }}>
              <img
                src="/assets/image/message-chat.svg"
                alt=""
                width="64"
                height="64"
                style={{ opacity: "0.5", marginBottom: "12px" }}
              />
              <p>No messages yet. Start the conversation!</p>
            </li>
          ) : (
            messages.map((message, index) => renderMessage(message, index))
          )}

          {/* Typing Indicator */}
          {isTyping && contact && (
            <li
              className="typing-indicator"
              style={{ textAlign: "left", padding: "10px 20px" }}
            >
              <div
                className="typing-dots"
                style={{ display: "flex", alignItems: "center", gap: "8px" }}
              >
                <span style={{ fontSize: "14px", color: "#666" }}>
                  {getDisplayName(contact)} is typing
                </span>
                <div style={{ display: "flex", gap: "2px" }}>
                  <div
                    className="typing-dot"
                    style={{
                      width: "6px",
                      height: "6px",
                      backgroundColor: "#007bff",
                      borderRadius: "50%",
                      animation: "typingBounce 1.4s infinite ease-in-out both",
                      animationDelay: "0s",
                    }}
                  ></div>
                  <div
                    className="typing-dot"
                    style={{
                      width: "6px",
                      height: "6px",
                      backgroundColor: "#007bff",
                      borderRadius: "50%",
                      animation: "typingBounce 1.4s infinite ease-in-out both",
                      animationDelay: "0.16s",
                    }}
                  ></div>
                  <div
                    className="typing-dot"
                    style={{
                      width: "6px",
                      height: "6px",
                      backgroundColor: "#007bff",
                      borderRadius: "50%",
                      animation: "typingBounce 1.4s infinite ease-in-out both",
                      animationDelay: "0.32s",
                    }}
                  ></div>
                </div>
              </div>
            </li>
          )}
        </ul>

        <div ref={messagesEndRef} />
      </div>

      {isProjectArchived ? (
        <div className="archived-project-notice">
          <div className="notice-content">
            <svg
              width="20"
              height="20"
              viewBox="0 0 20 20"
              fill="none"
              xmlns="http://www.w3.org/2000/svg"
            >
              <path
                d="M10 18.3333C14.6024 18.3333 18.3333 14.6024 18.3333 10C18.3333 5.39763 14.6024 1.66667 10 1.66667C5.39763 1.66667 1.66667 5.39763 1.66667 10C1.66667 14.6024 5.39763 18.3333 10 18.3333Z"
                stroke="#6B7280"
                strokeWidth="1.5"
                strokeLinecap="round"
                strokeLinejoin="round"
              />
              <path
                d="M10 6.66667V10"
                stroke="#6B7280"
                strokeWidth="1.5"
                strokeLinecap="round"
                strokeLinejoin="round"
              />
              <path
                d="M10 13.3333H10.0083"
                stroke="#6B7280"
                strokeWidth="1.5"
                strokeLinecap="round"
                strokeLinejoin="round"
              />
            </svg>
            <span>
              This project has been {projectStatus}. Messaging is no longer
              available.
              {projectStatus === "completed" && (
                <>
                  {" "}
                  You can now{" "}
                  <a
                    href={`/review/${projectId}`}
                    style={{ color: "#EF7310", textDecoration: "none" }}
                  >
                    leave a review
                  </a>
                  .
                </>
              )}
            </span>
          </div>
        </div>
      ) : (
        contact && (
          <MessageInput
            onSendMessage={sendMessage}
            onTyping={handleTyping}
            placeholder={`Message ${getDisplayName(contact)}...`}
            disabled={isProjectArchived}
          />
        )
      )}

      {selectedAttachment && (
        <AttachmentModal
          url={selectedAttachment.url}
          type={selectedAttachment.type}
          name={selectedAttachment.name}
          onClose={() => setSelectedAttachment(null)}
        />
      )}
    </div>
  );
}


