import { useState } from "react";
import Image from "next/image";
// import "./attachment-modal.css";

const AttachmentModal = ({ url, type, name, onClose }: { url: string, type: string, name?: string, onClose: () => void }) => {


  const handleClose = () => {
    onClose();
  };

  // Determine if it's an image or video
  const isImage = type.startsWith('image/') ||
    /\.(jpg|jpeg|png|gif|bmp|webp|svg)$/i.test(url);
  const isVideo = type.startsWith('video/') ||
    /\.(mp4|mov|avi|wmv|flv|webm|mkv)$/i.test(url);

  return (
    <div className="modal-overlay" onClick={handleClose}>
      <div className={`modal-content  ${isVideo ? 'pb-0' : ''}`} onClick={(e) => e.stopPropagation()}>
        <div className="modal-header">
          {/* <h5>{name || (isImage ? "Image" : isVideo ? "Video" : "File")}</h5> */}
          <button
            className="modal-close"
            onClick={handleClose}
          >
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M6 18L18 6M6 6L18 18" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
            </svg>
          </button>
        </div>

        <div className="modal-body">
          {isImage ? (
            <Image
              width={1920}
              height={1080}
              src={url}
              alt={name || "Image attachment"}
              className='img-fluid'
              style={{
                width: '100%',
                height: 'auto',
                maxHeight: '70vh',
                objectFit: 'contain'
              }}
            />
          ) : isVideo ? (
            <video
              controls
              autoPlay
              style={{
                width: '100%',
                height: '100%',
                outline: 'none'
              }}
            >
              <source src={url} type={type || 'video/mp4'} />
              Your browser does not support the video tag.
            </video>
          ) : null}
        </div>
      </div>

      <style jsx>{`
        .modal-overlay {
          position: fixed;
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background: rgba(0, 0, 0, 0.7);
          display: flex;
          align-items: center;
          justify-content: center;
          z-index: 9999;
          padding: 20px;
        }

        .modal-content {
          border-radius: 5px;
          width: 100%;
          max-width: 600px;
          max-height: 90vh;
          overflow-y: auto;
          animation: modalFadeIn 0.3s ease-out;
          position: relative;
          padding: 10px;
          background: var(--color-white);
        }

        @keyframes modalFadeIn {
          from {
            opacity: 0;
            transform: scale(0.9) translateY(-20px);
          }
          to {
            opacity: 1;
            transform: scale(1) translateY(0);
          }
        }

        .modal-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 0;
        }

        .modal-header h5 {
          font-size: var(--font-18);
          font-weight: 700;
          color: var(--typo-secondary);
          margin: 0;
        }

        .modal-close {
          background: none;
          border: none;
          padding: 0;
          cursor: pointer;
          color: var(--color-primary);
          transition: all .3s ease;
          position: absolute;
          right: 15px;
          top: 15px;
          z-index: 9;
          background-color: #ffffff;
          width: 30px;
          height: 30px;
          border-radius: 40px;
          box-shadow: rgba(0, 0, 0, 0.1) 0px 4px 12px;
          display: flex;
          align-items: center;
          justify-content: center;
        }

        .modal-close svg {
            width: 20px;
            height: 20px;
        }

        .modal-close:hover:not(:disabled) {
          color: var(--color-primary);
        }

        .modal-close:disabled {
          opacity: 0.5;
          cursor: not-allowed;
        }

        .upload-outer {
          margin-bottom: 20px;
        }

        .form-label {
          display: block;
          margin-bottom: 10px;
          font-weight: 600;
          font-size: var(--font-small);
          color: var(--typo-secondary);
        }

        .upload-area {
          position: relative;
          border: 2px dashed var(--border-color);
          border-radius: 12px;
          padding: 40px 20px;
          text-align: center;
          transition: all 0.3s ease;
          cursor: pointer;
          min-height: 200px;
          display: flex;
          align-items: center;
          justify-content: center;
        }

        .upload-area:hover {
          border-color: var(--color-primary);
          background: rgba(239, 115, 16, 0.02);
        }

        .preview-box {
          position: relative;
          display: inline-block;
        }

        .preview-image {
          border-radius: 12px;
          object-fit: cover;
          max-width: 200px;
          max-height: 200px;
        }

        .remove-btn {
          position: absolute;
          top: -10px;
          right: -10px;
          width: 32px;
          height: 32px;
          background: #e74c3c;
          color: white;
          border: none;
          border-radius: 50%;
          cursor: pointer;
          display: flex;
          align-items: center;
          justify-content: center;
          font-size: 18px;
          font-weight: bold;
          transition: all 0.3s ease;
        }

        .remove-btn:hover:not(:disabled) {
          background: #c0392b;
          transform: scale(1.1);
        }

        .remove-btn:disabled {
          opacity: 0.5;
          cursor: not-allowed;
        }

        .upload-placeholder {
          display: flex;
          flex-direction: column;
          align-items: center;
          gap: 15px;
        }

        .upload-placeholder p {
          font-size: var(--font-small);
          color: var(--typo-secondary);
          margin: 0;
        }

        .browse-btn {
          background: none;
          border: none;
          color: var(--color-primary);
          text-decoration: underline;
          cursor: pointer;
          font-size: inherit;
          padding: 0;
        }

        .browse-btn:hover:not(:disabled) {
          opacity: 0.8;
        }

        .browse-btn:disabled {
          opacity: 0.5;
          cursor: not-allowed;
        }

        .text-muted {
          color: var(--text-color);
          font-size: var(--font-xs);
        }

        .file-input {
          display: none;
        }

        .modal-actions {
          display: flex;
          justify-content: flex-end;
          gap: 15px;
          margin-top: 25px;
          padding-top: 20px;
          border-top: 1px solid var(--border-color);
        }

        .custom-btn {
          display: inline-flex;
          align-items: center;
          justify-content: center;
          padding: 10px 20px;
          border-radius: 8px;
          font-size: 16px;
          font-weight: 600;
          text-decoration: none;
          cursor: pointer;
          transition: all 0.3s ease;
          border: 1px solid transparent;
          min-width: 100px;
        }

        .custom-btn.primary-fill {
          background: var(--color-primary);
          color: var(--color-white);
          border-color: var(--color-primary);
        }

        .custom-btn.primary-fill:hover:not(:disabled) {
          background: transparent;
          color: var(--color-primary);
        }

        .custom-btn.secondary-outline {
          background: transparent;
          color: var(--text-color);
          border-color: var(--border-color);
        }

        .custom-btn.secondary-outline:hover:not(:disabled) {
          background: var(--border-color);
          color: var(--typo-secondary);
        }

        .custom-btn:disabled {
          opacity: 0.6;
          cursor: not-allowed;
        }

        @media (max-width: 576px) {
          .modal-content {
            margin: 10px;
            max-height: calc(100vh - 20px);
          }

          .upload-area {
            padding: 30px 15px;
          }

          .modal-actions {
            flex-direction: column;
            gap: 10px;
          }

          .custom-btn {
            width: 100%;
          }
        }
      `}</style>
    </div>
  );
}

export default AttachmentModal;