import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/auth";
import { prisma } from "@/lib/prisma";
import bcrypt from "bcryptjs";

// export async function GET(req: NextRequest) {
//   try {
//     // Check if required dependencies are available
//     if (!prisma) {
//       return NextResponse.json(
//         { error: "Database connection not available" },
//         { status: 503 }
//       );
//     }
//     const session = await getServerSession(authOptions);

//     if (!session?.user?.email) {
//       return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
//     }
//     const userId = session?.user?.id; // or any user ID you have

//     const conversations = await prisma.conversation.findMany({
//       where: {
//         OR: [{ buyerId: userId }, { sellerId: userId }],
//       },
//       distinct: ["projectId"],
//       include: {
//         // project: true,      // optional: include related project details
//         buyer: {
//           select: {
//             id: true,
//             email: true,
//             profile: {
//               select: {
//                 firstName: true,
//                 lastName: true,
//                 profileImage: true,
//               },
//             },
//           },
//         },
//         seller: {
//           select: {
//             id: true,
//             email: true,
//             profile: {
//               select: {
//                 firstName: true,
//                 lastName: true,
//                 profileImage: true,
//               },
//             },
//           },
//         },
//         // chats: {
//         //   take: 1,
//         //   orderBy: { createdAt: 'desc' }, // get latest message
//         // },
//       },
//       orderBy: {
//         updatedAt: "desc", // show most recent conversations first
//       },
//     });

//     // Then get conversations grouped by project
//     const conversationsByProject = await Promise.all(
//       conversations.map(async (userProject) => {
//         const conversations = await prisma.conversation.findMany({
//           where: {
//             projectId: userProject.projectId,
//             OR: [{ buyerId: userId }, { sellerId: userId }],
//           },
//           include: {
//             buyer: {
//               select: {
//                 id: true,
//                 email: true,
//                 profile: {
//                   select: {
//                     firstName: true,
//                     lastName: true,
//                     profileImage: true,
//                   },
//                 },
//               },
//             },
//             seller: {
//               select: {
//                 id: true,
//                 email: true,
//                 profile: {
//                   select: {
//                     firstName: true,
//                     lastName: true,
//                     profileImage: true,
//                   },
//                 },
//               },
//             },
//             // chats: {
//             //   take: 1,
//             //   orderBy: { createdAt: 'desc' },
//             // },
//           },
//           orderBy: {
//             updatedAt: "desc",
//           },
//         });

//         return {
//           projectId: userProject.projectId,
//           project: userProject.productSnapshot,
//           conversations: conversations,
//         };
//       })
//     );

//     return NextResponse.json({
//       message: "Conversation created successfully",
//       // conversationData: conversations,
//       conversationData: conversationsByProject,
//     });
//   } catch (dbError) {
//     console.error("Database error during password change:", dbError);
//     throw dbError;
//   }
// }


// export async function GET(req: NextRequest) {
//   try {
//     if (!prisma) {
//       return NextResponse.json(
//         { error: "Database connection not available" },
//         { status: 503 }
//       );
//     }
//     const session = await getServerSession(authOptions);

//     if (!session?.user?.email) {
//       return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
//     }
//     const userId = session?.user?.id;

//     const { searchParams } = new URL(req.url);
//     const page = parseInt(searchParams.get('page') || '1');
//     const limit = parseInt(searchParams.get('limit') || '50');
//     const skip = (page - 1) * limit;

//     const conversationId = searchParams.get('conversationId');

//     if (conversationId) {
//       const conversation = await prisma.conversation.findFirst({
//         where: {
//           id: conversationId,
//           OR: [{ buyerId: userId }, { sellerId: userId }], 
//         },
//         include: {
//           buyer: {
//             select: {
//               id: true,
//               email: true,
//               profile: {
//                 select: {
//                   firstName: true,
//                   lastName: true,
//                   profileImage: true,
//                 },
//               },
//             },
//           },
//           seller: {
//             select: {
//               id: true,
//               email: true,
//               profile: {
//                 select: {
//                   firstName: true,
//                   lastName: true,
//                   profileImage: true,
//                 },
//               },
//             },
//           },
//         },
//       });

//       if (!conversation) {
//         return NextResponse.json(
//           { error: "Conversation not found" },
//           { status: 404 }
//         );
//       }

//       const projectConversations = await prisma.conversation.findMany({
//         where: {
//           projectId: conversation.projectId,
//           OR: [{ buyerId: userId }, { sellerId: userId }],
//         },
//         include: {
//           buyer: {
//             select: {
//               id: true,
//               email: true,
//               profile: {
//                 select: {
//                   firstName: true,
//                   lastName: true,
//                   profileImage: true,
//                 },
//               },
//             },
//           },
//           seller: {
//             select: {
//               id: true,
//               email: true,
//               profile: {
//                 select: {
//                   firstName: true,
//                   lastName: true,
//                   profileImage: true,
//                 },
//               },
//             },
//           },
//         },
//         orderBy: {
//           updatedAt: "desc",
//         },
//       });

//       const conversationByProject = {
//         projectId: conversation.projectId,
//         project: conversation.productSnapshot,
//         conversations: projectConversations,
//       };

//       return NextResponse.json({
//         message: "Conversation fetched successfully",
//         conversationData: conversationByProject, 
//         pagination: {
//           currentPage: 1,
//           totalPages: 1,
//           totalItems: 1,
//           hasNext: false,
//           hasPrev: false,
//           limit: 1
//         }
//       });
//     }

//     const totalConversations = await prisma.conversation.count({
//       where: {
//         OR: [{ buyerId: userId }, { sellerId: userId }],
//       },
//     });

//     const conversations = await prisma.conversation.findMany({
//       where: {
//         OR: [{ buyerId: userId }, { sellerId: userId }],
//       },
//       distinct: ["projectId"],
//       include: {
//         buyer: {
//           select: {
//             id: true,
//             email: true,
//             profile: {
//               select: {
//                 firstName: true,
//                 lastName: true,
//                 profileImage: true,
//               },
//             },
//           },
//         },
//         seller: {
//           select: {
//             id: true,
//             email: true,
//             profile: {
//               select: {
//                 firstName: true,
//                 lastName: true,
//                 profileImage: true,
//               },
//             },
//           },
//         },
//       },
//       orderBy: {
//         updatedAt: "desc",
//       },
//       skip: skip,
//       take: limit,
//     });


//     const conversationsByProject = await Promise.all(
//       conversations.map(async (userProject) => {
//         const conversations = await prisma.conversation.findMany({
//           where: {
//             projectId: userProject.projectId,
//             OR: [{ buyerId: userId }, { sellerId: userId }],
//           },
//           include: {
//             buyer: {
//               select: {
//                 id: true,
//                 email: true,
//                 profile: {
//                   select: {
//                     firstName: true,
//                     lastName: true,
//                     profileImage: true,
//                   },
//                 },
//               },
//             },
//             seller: {
//               select: {
//                 id: true,
//                 email: true,
//                 profile: {
//                   select: {
//                     firstName: true,
//                     lastName: true,
//                     profileImage: true,
//                   },
//                 },
//               },
//             },
//           },
//           orderBy: {
//             updatedAt: "desc",
//           },
//         });

//         return {
//           projectId: userProject.projectId,
//           project: userProject.productSnapshot,
//           conversations: conversations,
//         };
//       })
//     );

//     const totalPages = Math.ceil(totalConversations / limit);
//     const hasNext = page < totalPages;
//     const hasPrev = page > 1;

//     return NextResponse.json({
//       message: "Conversations fetched successfully",
//       conversationData: conversationsByProject,
//       pagination: {
//         currentPage: page,
//         totalPages: totalPages,
//         totalItems: totalConversations,
//         hasNext: hasNext,
//         hasPrev: hasPrev,
//         limit: limit
//       }
//     });
//   } catch (dbError) {
//     console.error("Database error during conversation fetch:", dbError);
//     return NextResponse.json(
//       { error: "Internal server error" },
//       { status: 500 }
//     );
//   }
// }


export async function GET(req: NextRequest) {
  try {
    if (!prisma) {
      return NextResponse.json(
        { error: "Database connection not available" },
        { status: 503 }
      );
    }

    const session = await getServerSession(authOptions);
    if (!session?.user?.email) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    const userId = session.user.id;
    const { searchParams } = new URL(req.url);
    const page = parseInt(searchParams.get("page") || "1");
    const limit = parseInt(searchParams.get("limit") || "50");
    const skip = (page - 1) * limit;

    const conversationId = searchParams.get("conversationId");

    
    if (conversationId) {
      const conversation = await prisma.conversation.findFirst({
        where: {
          id: conversationId,
          OR: [{ buyerId: userId }, { sellerId: userId }],
        },
        include: {
          buyer: {
            select: {
              id: true,
              email: true,
              profile: { select: { firstName: true, lastName: true, profileImage: true } },
            },
          },
          seller: {
            select: {
              id: true,
              email: true,
              profile: { select: { firstName: true, lastName: true, profileImage: true } },
            },
          },
        },
      });

      if (!conversation) {
        return NextResponse.json({ error: "Conversation not found" }, { status: 404 });
      }

      
      const userUnreadCount = await prisma.message.count({
        where: {
          conversationId: conversation.id,
          receiverId: userId,
          isRead: false,
        },
      });

      
      const projectConversations = await prisma.conversation.findMany({
        where: {
          projectId: conversation.projectId,
          OR: [{ buyerId: userId }, { sellerId: userId }],
        },
        include: {
          buyer: { select: { id: true, email: true, profile: true } },
          seller: { select: { id: true, email: true, profile: true } },
        },
        orderBy: { updatedAt: "desc" },
      });

      
      const projectUnreadCount = await prisma.message.count({
        where: {
          conversationId: { in: projectConversations.map((c: any) => c.id) },
          receiverId: userId,
          isRead: false,
        },
      });

      
      const totalUnreadCount = await prisma.message.count({
        where: { receiverId: userId, isRead: false },
      });

      const conversationByProject = {
        projectId: conversation.projectId,
        project: conversation.productSnapshot,
        conversations: projectConversations,
        unreadCount: projectUnreadCount,
        hasUnreadMessages: projectUnreadCount > 0,
      };

      return NextResponse.json({
        message: "Conversation fetched successfully",
        conversationData: conversationByProject,
        unreadCounts: {
          currentConversation: userUnreadCount,
          projectUnreadCount,
          totalUnreadCount,
        },
      });
    }

    const totalConversations = await prisma.conversation.count({
      where: {
        OR: [{ buyerId: userId }, { sellerId: userId }],
      },
    });

    const conversations = await prisma.conversation.findMany({
      where: {
        OR: [{ buyerId: userId }, { sellerId: userId }],
      },
      distinct: ["projectId"],
      include: {
        buyer: { select: { id: true, email: true, profile: true } },
        seller: { select: { id: true, email: true, profile: true } },
      },
      orderBy: { updatedAt: "desc" },
      skip,
      take: limit,
    });

    const conversationsByProject = await Promise.all(
      conversations.map(async (projectConversation: any) => {
        const projectConversations = await prisma.conversation.findMany({
          where: {
            projectId: projectConversation.projectId,
            OR: [{ buyerId: userId }, { sellerId: userId }],
          },
          include: {
            buyer: { select: { id: true, email: true, profile: true } },
            seller: { select: { id: true, email: true, profile: true } },
          },
          orderBy: { updatedAt: "desc" },
        });

        const unreadCount = await prisma.message.count({
          where: {
            conversationId: { in: projectConversations.map((c: any) => c.id) },
            receiverId: userId,
            isRead: false,
          },
        });

        return {
          projectId: projectConversation.projectId,
          project: projectConversation.productSnapshot,
          conversations: projectConversations,
          unreadCount,
          hasUnreadMessages: unreadCount > 0,
        };
      })
    );

    const userTotalUnreadCount = await prisma.message.count({
      where: { receiverId: userId, isRead: false },
    });

    const totalPages = Math.ceil(totalConversations / limit);
    const hasNext = page < totalPages;
    const hasPrev = page > 1;

    return NextResponse.json({
      message: "Conversations fetched successfully",
      conversationData: conversationsByProject,
      unreadCounts: {
        totalUnreadCount: userTotalUnreadCount,
        byProject: conversationsByProject.map((p: any) => ({
          projectId: p.projectId,
          unreadCount: p.unreadCount,
          hasUnreadMessages: p.hasUnreadMessages,
        })),
      },
      pagination: {
        currentPage: page,
        totalPages,
        totalItems: totalConversations,
        hasNext,
        hasPrev,
        limit,
      },
    });
  } catch (err) {
    console.error("Error fetching conversations:", err);
    return NextResponse.json({ error: "Internal server error" }, { status: 500 });
  }
}

