import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function POST(req: NextRequest) {
  try {
    // Check if required dependencies are available
    if (!prisma) {
      return NextResponse.json(
        { error: 'Database connection not available' },
        { status: 503 }
      );
    }

    const session = await getServerSession(authOptions);
    
    if (!session?.user?.email) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }

    const body = await req.json();
    const { projectId, buyerId, sellerId, lastMessage} = body;

    if (!projectId || !buyerId || !sellerId) {
      return NextResponse.json({ error: "Missing required fields" }, { status: 400 });
    }
    
    // 1️⃣ Check if conversation already exists
        const existingConversation = await prisma.conversation.findFirst({
            where: {
                projectId,
                OR: [
                    { buyerId, sellerId },
                    { buyerId: sellerId, sellerId: buyerId }, // handle reversed roles just in case
                ],
            },
        });
        // 2️⃣ If exists → return it
        if (existingConversation) {
            return NextResponse.json({
                message: "Conversation already exists",
                conversationId: existingConversation.id,
            });
        }

        const projectData  = await prisma.project.findFirst({
            where: {id: projectId },
        });
        if (!projectData) {
            return NextResponse.json({ error: "Project is Missing" }, { status: 400 });
        }
        

       const newConversation  = await prisma.conversation.create({
            data: {
            projectId,
            buyerId,
            sellerId,
            productSnapshot:projectData,
            lastMessage
            },
        });
        return NextResponse.json({
            message: "Conversation created successfully",
            conversationId: newConversation.id,
        });   
    } catch (dbError) {
      console.error('Database error during password change:', dbError);
      throw dbError;
    }

  }