import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import bcrypt from 'bcryptjs';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { verifyJwtToken } from '@/lib/api/jwt';


export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { token, password } = body;

    if (!token || !password) {
      return NextResponse.json(
        { message: 'Token and new password are required' },
        { status: 400 }
      );
    }

    // verify token
    const decoded = await verifyJwtToken({
      token,
      secret: process.env.JWT_SECRET_KEY as string
    });

    if (!decoded || !decoded.id) {
      return NextResponse.json(
        { message: 'Invalid or expired token' },
        { status: 400 }
      );
    }

    // find user by ID
    const existingUser = await prisma.user.findUnique({
      where: { id: decoded.id as string }
    });

    if (!existingUser) {
      return NextResponse.json(
        { message: 'User not found' },
        { status: 404 }
      );
    }

    // ensure the token matches the one stored in DB (prevent reuse / forgery)
    if (existingUser.passwordResetToken !== token) {
      console.log('Existing: ', existingUser.passwordResetToken, ' MY: ', token);
      return NextResponse.json(
        { message: 'Token mismatch or already used' },
        { status: 400 }
      );
    }

    // hash the new password
    const hashedNewPassword = await bcrypt.hash(password, 12);

    // update password and clear reset token fields
    await prisma.user.update({
      where: { id: existingUser.id },
      data: {
        password: hashedNewPassword,
        passwordResetToken: null,
        passwordResetAt: null
      }
    });

    return NextResponse.json(
      { message: 'Password has been reset successfully' },
      { status: 200 }
    );
  } catch (error) {
    console.error('Change password error:', error);
    return NextResponse.json(
      { message: 'Internal server error' },
      { status: 500 }
    );
  }
}