"use client";

import { useState, useEffect, useRef } from "react";
import { useSession } from "next-auth/react";
import { useWebSocketContext } from "@/context/WebSocketContext";

export default function ConversationList({
  onSelectConversation,
  selectedContactId,
  convId,
  onFirstConversationAvailable,
  shouldSelectFirst,
  showSidebar,
  selectedContact
}: any) {
  const { data: session } = useSession();
  const [conversations, setConversations] = useState<any>([]);
  const [loading, setLoading] = useState(false);
  const [query, setQuery] = useState("");

  const [page, setPage] = useState(1);
  const [hasMore, setHasMore] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);

  // make a copy to maintain filtered list
  const [filteredConversations, setFilteredConversations] = useState<any>([]);

  const { isConnected, lastMessage } = useWebSocketContext();

  const containerRef = useRef<HTMLDivElement>(null);
  const isInitialLoadRef = useRef(true);

  const [showMore, setShowMore] = useState<boolean>(true);

  useEffect(() => {
    if (session?.user?.email) {
      fetchConversations(1, false);
    }
  }, []);

  const fetchConversations = async (pageNum: number = 1, append: boolean = false) => {
    try {
      if (append) {
        setLoadingMore(true);
      } else {
        setLoading(true);
      }

      const response = await fetch(`/api/chat_conversation_sidebar?page=${pageNum}&limit=10`);
      if (response.ok) {
        const data = await response.json();

        if (append) {
          setConversations((prev: any) => {
            const existingIds = new Set(prev.map((conv: any) => conv.projectId));
            const newConversations = data.conversationData.filter(
              (conv: any) => !existingIds.has(conv.projectId)
            );
            return [...prev, ...newConversations];
          });

          setFilteredConversations((prev: any) => {
            const existingIds = new Set(prev.map((conv: any) => conv.projectId));
            const newConversations = data.conversationData.filter(
              (conv: any) => !existingIds.has(conv.projectId)
            );
            return [...prev, ...newConversations];
          });
        } else {
          setConversations(data.conversationData);
          setFilteredConversations(data.conversationData);
        }

        // has more?
        setHasMore(data?.pagination?.hasNext);
      }
    } catch (error) {
      console.error("Error fetching conversations:", error);
    } finally {
      setLoading(false);
      setLoadingMore(false);
    }
  };

  const loadMoreConversations = () => {
    if (!loadingMore && hasMore) {
      const nextPage = page + 1;
      setPage(nextPage);
      fetchConversations(nextPage, true);
    }
  };

  useEffect(() => {
    const container = containerRef.current;
    if (!container) return;

    const handleScroll = () => {
      const { scrollTop, scrollHeight, clientHeight } = container;
      // Load more when near bottom (within 100px)
      if (scrollHeight - scrollTop - clientHeight < 100 && hasMore && !loadingMore && !loading) {
        loadMoreConversations();
      }
    };

    container.addEventListener('scroll', handleScroll);
    return () => container.removeEventListener('scroll', handleScroll);
  }, [hasMore, loadingMore, loading]);


  useEffect(() => {
    if (!lastMessage) return;

    try {
      const message = typeof lastMessage === 'string' ? JSON.parse(lastMessage) : lastMessage;

      if (message.type === 'new_message') {
        handleNewMessage(message);
      } else if (message.type === 'conversation_read') {
        handleConversationRead(message);
      } else if(message.type === "message_read") {
        const { conversationId, projectId, senderId } = message.message;

        const receiverId = selectedContact?.id;
        if(!receiverId) return;


        // this project should have zero unReadcount if the selectedContactId matches
        if ( senderId === receiverId && projectId && conversationId && convId === conversationId) {
          setConversations((prev: any[]) => {
            return prev.map((conversation: any) => {
              if (conversation.projectId === projectId) {
                return {
                  ...conversation,
                  hasUnreadMessages: false,
                  unreadCount: 0
                };
              }
              return conversation;
            });
          });

          setFilteredConversations((prev: any[]) => {
            return prev.map((conversation: any) => {
              if (conversation.projectId === projectId) {
                return {
                  ...conversation,
                  hasUnreadMessages: false,
                  unreadCount: 0
                };
              }
              return conversation;
            });
          });
        }


      }
    } catch (error) {
      console.error("Error processing WebSocket message:", error);
    }
  }, [lastMessage]);

  const handleConversationRead = async (message:any) => {
    if (!message.message) return;

    const { projectId, count: finalCnt } = message.message;

        setConversations((prev: any[]) => {
          return prev.map((conversation: any) => {
            if (conversation.projectId === projectId) {
              const newUnreadCount = Math.max(0, (conversation.unreadCount || 0) - finalCnt);
              return {
                ...conversation,
                hasUnreadMessages: newUnreadCount > 0,
                unreadCount: newUnreadCount
              };
            }
            return conversation;
          });
        });

        setFilteredConversations((prev: any[]) => {
          return prev.map((conversation: any) => {
            if (conversation.projectId === projectId) {
              const newUnreadCount = Math.max(0, (conversation.unreadCount || 0) - finalCnt);
              return {
                ...conversation,
                hasUnreadMessages: newUnreadCount > 0,
                unreadCount: newUnreadCount
              };
            }
            return conversation;
          });
        });
  }

  const handleNewMessage = async (message: any) => {
    if (!message.message) return;

    const msg = message.message;
    const projectId = msg.projectId;

    try {
      const response = await fetch(`/api/chat_conversation_sidebar?conversationId=${msg.conversationId}`);
      if (!response.ok) return;

      const data = await response.json();
      const newConversation = data.conversationData;


      setConversations((prev: any) => {
        const exists = prev.some((conv: any) => conv.projectId === newConversation.projectId);
        if (exists) {

          const updated = prev.map((conv: any) =>
            conv.projectId === newConversation.projectId ? newConversation : conv
          );

          const withoutOld = updated.filter((c: any) => c.projectId !== newConversation.projectId);
          return [newConversation, ...withoutOld];
        }
        return [newConversation, ...prev];
      });


      setFilteredConversations((prev: any) => {
        const exists = prev.some((conv: any) => conv.projectId === newConversation.projectId);
        if (exists) {
          const updated = prev.map((conv: any) =>
            conv.projectId === newConversation.projectId ? newConversation : conv
          );
          const withoutOld = updated.filter((c: any) => c.projectId !== newConversation.projectId);
          return [newConversation, ...withoutOld];
        }
        return [newConversation, ...prev];
      });
    } catch (err) {
      console.error("Error updating conversation sidebar:", err);
    }
  };


  const formatTimeAgo = (dateString: string) => {
    const date = new Date(dateString);
    const now = new Date();
    const diffInMinutes = Math.floor(
      (now.getTime() - date.getTime()) / (1000 * 60)
    );

    if (diffInMinutes < 1) return "Just now";
    if (diffInMinutes < 60) return `${diffInMinutes}m ago`;
    if (diffInMinutes < 1440) return `${Math.floor(diffInMinutes / 60)}h ago`;
    return `${Math.floor(diffInMinutes / 1440)}d ago`;
  };
  const getDisplayName = (user: any) => {
    if (user.profile?.firstName && user.profile?.lastName) {
      return `${user.profile.firstName} ${user.profile.lastName}`;
    }
    return user.email;
  };
  const getProfileImage = (user: any) => {
    return user.profile?.profileImage || "/assets/image/harveyspecter.png";
  };
  const truncateMessage = (content: string, maxLength: number = 50) => {
    if (content.length <= maxLength) return content;
    return content.substring(0, maxLength) + "...";
  };

  const handleSearch = (searchText: string) => {
    setQuery(searchText);

    if (!searchText.trim()) {
      setFilteredConversations(conversations);
      return;
    }

    const lowerQuery = searchText.toLowerCase();

    const filtered = conversations.filter((conversation: any) => {
      const projectTitle = conversation?.project?.title?.toLowerCase() || "";
      return projectTitle.includes(lowerQuery);
    });

    setFilteredConversations(filtered);
  };


  if (!session?.user) {
    return null;
  }

  return (
    <div id="sidepanel" className={`sidebar-chat ${!showSidebar ? "sidebar-chat-hidden" : ""}`}>
      <div className="search-wrap" id="search">
        <label htmlFor="search-input">
          <svg
            width="20"
            height="20"
            viewBox="0 0 20 20"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
          >
            <path
              d="M19.5308 18.4686L14.8368 13.7755C16.1973 12.1421 16.8757 10.047 16.7309 7.92618C16.5861 5.80531 15.6293 3.82191 14.0593 2.38859C12.4894 0.955257 10.4274 0.18235 8.30212 0.23065C6.17687 0.27895 4.15205 1.14474 2.64888 2.64791C1.14571 4.15108 0.279927 6.1759 0.231627 8.30115C0.183327 10.4264 0.956234 12.4885 2.38956 14.0584C3.82289 15.6283 5.80629 16.5852 7.92715 16.73C10.048 16.8748 12.1431 16.1963 13.7765 14.8358L18.4696 19.5299C18.5393 19.5995 18.622 19.6548 18.713 19.6925C18.8041 19.7302 18.9017 19.7497 19.0002 19.7497C19.0988 19.7497 19.1963 19.7302 19.2874 19.6925C19.3784 19.6548 19.4612 19.5995 19.5308 19.5299C19.6005 19.4602 19.6558 19.3775 19.6935 19.2864C19.7312 19.1954 19.7506 19.0978 19.7506 18.9992C19.7506 18.9007 19.7312 18.8031 19.6935 18.7121C19.6558 18.621 19.6005 18.5383 19.5308 18.4686ZM1.75021 8.49924C1.75021 7.16421 2.14609 5.85917 2.88779 4.74914C3.62949 3.63911 4.6837 2.77394 5.9171 2.26305C7.1505 1.75216 8.5077 1.61849 9.81707 1.87894C11.1264 2.13939 12.3292 2.78226 13.2732 3.72627C14.2172 4.67027 14.8601 5.87301 15.1205 7.18238C15.381 8.49175 15.2473 9.84895 14.7364 11.0823C14.2255 12.3158 13.3603 13.37 12.2503 14.1117C11.1403 14.8534 9.83524 15.2492 8.50021 15.2492C6.71061 15.2473 4.99488 14.5355 3.72944 13.27C2.46399 12.0046 1.7522 10.2888 1.75021 8.49924Z"
              fill="#404A62"
            />
          </svg>
        </label>
        <input
          type="text"
          id="search-input"
          placeholder="Search for chats..."
          value={query}
          onChange={(e) => handleSearch(e.target.value)}
        />
      </div>

      <div id="contacts" ref={containerRef} style={{ overflowY: 'auto', height: '100%' }}>
        {loading ? (
          <div className="p-4 text-center">
            <div className="spinner-border spinner-border-sm" role="status">
              <span className="visually-hidden">Loading...</span>
            </div>
          </div>
        ) : filteredConversations.length === 0 ? (
          <div className="p-4 text-center text-muted">
            <img
              src="/assets/image/message-chat.svg"
              alt=""
              width="48"
              height="48"
              className="mb-2 opacity-50"
            />
            <p className="mb-0">No conversations found</p>
          </div>
        ) : (
          <ul>
            {filteredConversations
              // .sort((a: any, b: any) => {
              //   const timeA = new Date(a.lastMessageTime || a.conversations[0]?.lastMessageTime || a.createdAt).getTime();
              //   const timeB = new Date(b.lastMessageTime || b.conversations[0]?.lastMessageTime || b.createdAt).getTime();
              //   return timeB - timeA; 
              // })
              .map((conversation: any) => (
                <li
                  key={conversation.conversations[0]?.id}
                  className={`contact sidebar-conversation-item ${conversation.conversations.some(
                    (conv: any) => conv.id === convId
                  ) ? "active" : ""
                    }`}
                  onClick={() => {
                    let user =
                      session?.user?.id ===
                        conversation.conversations[0]?.buyer?.id
                        ? conversation.conversations[0]?.seller
                        : conversation.conversations[0]?.buyer;

                    onSelectConversation(
                      conversation.conversations[0]?.id,
                      user,
                      conversation.project
                    );
                  }}
                >
                  <div className="sidebar-conversation-item-list">
                    {conversation.conversations.length - 2 > 0 && (
                      <div
                        className="show-more-bid-users"
                        onClick={(e) => {
                          e.stopPropagation();
                          setShowMore((prev) => !prev)
                        }}
                      >
                        {showMore ? `+${conversation.conversations.length - 2} more` : "Show less"}
                      </div>
                    )}
                    { conversation.hasUnreadMessages && (
                      <span className={`message-count ${conversation.conversations.length - 2 > 0 ? "message-count-more" : ""}`}>{conversation.unreadCount}</span>
                    )}
                    <div className="meta project-chat-meta">
                      {conversation.project && (
                        <p className="name">{conversation?.project?.title}</p>
                      )}
                    </div>
                    <div className="w-100 show-bid-users-list">
                      <ul className={`${showMore ? "bid-user-list-wrap" : ""}`}>
                        {conversation.conversations?.map(
                          (userlist: any, ind: any) => (
                            <li
                              key={ind}
                              onClick={(e) => {
                                e.stopPropagation();
                                let user =
                                  session?.user?.id === userlist?.buyer?.id
                                    ? userlist?.seller
                                    : userlist?.buyer;
                                onSelectConversation(
                                  userlist?.id,
                                  user,
                                  userlist?.productSnapshot
                                );
                              }}
                              className="bid-user-list-item"
                            >
                              <div className="d-flex align-items-center chat-user-list">
                                <div className="wrap relative">
                                  <span className="contact-status online"></span>
                                  <img src="/assets/image/harveyspecter.png" />
                                </div>
                                <div className="user-name-row">

                                  {/* {
                                  session?.user?.id === userlist?.buyer?.id ?( 
                                    {userlist?.buyer?.profile?.firstName}{" "}
                                   {userlist?.buyer?.profile?.lastName}

                                   ):(
                                       {userlist?.seller?.profile?.firstName}{" "}
                                        {userlist?.seller?.profile?.lastName}
                                   )
                                } */}
                                  {
                                    session?.user?.id === userlist?.buyer?.id ? (
                                      <>
                                        {userlist?.seller?.profile?.firstName} {userlist?.seller?.profile?.lastName}
                                      </>
                                    ) : (
                                      <>
                                        {userlist?.buyer?.profile?.firstName} {userlist?.buyer?.profile?.lastName}
                                      </>
                                    )
                                  }
                                </div>
                              </div>
                            </li>
                          )
                        )}
                      </ul>
                    </div>
                  </div>
                </li>
              ))}

            {loadingMore && (
              <li className="p-3 text-center">
                <div className="spinner-border spinner-border-sm" role="status">
                  <span className="visually-hidden">Loading more...</span>
                </div>
              </li>
            )}

            {/* ADD END OF LIST MESSAGE */}
            {!hasMore && conversations.length > 0 && (
              <li className="p-3 text-center text-muted">
                <small>No more conversations to load</small>
              </li>
            )}
          </ul>
        )}
      </div>
    </div>
  );
}
