'use client';

import { useState, useEffect } from 'react';

interface ReviewStatusProps {
  projectId: string;
  userRole: 'freelancer' | 'client';
}

interface ReviewStatus {
  hasReviewed: boolean;
  reviewId?: string;
  rating?: number;
  comment?: string;
  createdAt?: string;
}

export default function ReviewStatus({ projectId, userRole }: ReviewStatusProps) {
  const [reviewStatus, setReviewStatus] = useState<ReviewStatus | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    if (projectId) {
      fetchReviewStatus();
    }
  }, [projectId]);

  const fetchReviewStatus = async () => {
    try {
      const response = await fetch(`/api/projects/${projectId}/review`);
      
      if (response.ok) {
        const data = await response.json();
        
        if (data.success && data.data) {
          // Find review by current user
          const userReview = data.data.find((review: any) => 
            review.reviewerRole === userRole
          );
          
          if (userReview) {
            setReviewStatus({
              hasReviewed: true,
              reviewId: userReview.id,
              rating: userReview.rating,
              comment: userReview.comment,
              createdAt: userReview.createdAt
            });
          } else {
            setReviewStatus({ hasReviewed: false });
          }
        } else {
          setReviewStatus({ hasReviewed: false });
        }
      } else {
        setReviewStatus({ hasReviewed: false });
      }
    } catch (error) {
      console.error('Error fetching review status:', error);
      setReviewStatus({ hasReviewed: false });
    } finally {
      setIsLoading(false);
    }
  };

  if (isLoading) {
    return (
      <div className="review-status-loading">
        <small className="text-muted">Checking review status...</small>
      </div>
    );
  }

  if (!reviewStatus) {
    return null;
  }

  if (reviewStatus.hasReviewed) {
    return (
      <div className="review-status-completed">
        <div className="d-flex align-items-center gap-2">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" stroke="#28a745" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"/>
          </svg>
          <small className="text-success fw-medium">Review Submitted</small>
        </div>
        {reviewStatus.rating && (
          <div className="review-preview mt-1">
            <div className="d-flex align-items-center gap-1">
              {[1, 2, 3, 4, 5].map((star) => (
                <svg
                  key={star}
                  width="12"
                  height="12"
                  viewBox="0 0 24 24"
                  fill={star <= reviewStatus.rating! ? "#FB6107" : "none"}
                  stroke="#FB6107"
                  strokeWidth="1"
                >
                  <path d="M12 .587l3.668 7.571 8.332 1.151-6.064 5.845 1.528 8.263L12 18.896l-7.464 4.521 1.528-8.263L.001 9.309l8.332-1.151z"/>
                </svg>
              ))}
              <small className="text-muted ms-1">
                {reviewStatus.createdAt && new Date(reviewStatus.createdAt).toLocaleDateString()}
              </small>
            </div>
          </div>
        )}
      </div>
    );
  }

  return (
    <div className="review-status-pending">
      <div className="d-flex align-items-center gap-2">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" stroke="#ffc107" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"/>
        </svg>
        <small className="text-warning">Review Pending</small>
      </div>
    </div>
  );
}