'use client';

import Link from 'next/link';
import Image from 'next/image';
import { usePathname } from 'next/navigation';
import { useSession, signOut } from 'next-auth/react';
import { useState } from 'react';
import NotificationPanel from './notifications/NotificationPanel';
import MessagesDropdown from './messages/MessagesDropdown';
import { useRememberMe } from '@/hooks/useRememberMe';
import { useTranslations } from 'next-intl';

interface HeaderProps {
  locale: string;
}

export default function Header({ locale }: HeaderProps) {
  const pathname = usePathname();
  const { data: session, status } = useSession();
  const { clearEmail } = useRememberMe();
  console.log("session ====", session);


  const handleSignOut = async () => {
    // Clear remembered email when user explicitly logs out
    clearEmail();
    await signOut({ callbackUrl: `/${locale}` });
  };

  const userRole = (session?.user as any)?.role;
  const isClient = userRole === 'CLIENT';

  const t = useTranslations();

  return (
    <header>
      <nav className="navbar navbar-expand-lg">
        <div className="container-fluid">
          <Link className="navbar-brand p-0" href={`/${locale}`}>
            <Image 
              src="/assets/image/logo.png" 
              className="img-fluid" 
              alt="logo"
              width={128}
              height={59}
              priority
            />
          </Link>
          
          <div className="collapse navbar-collapse" id="navbarSupportedContent">
            <ul className="navbar-nav me-auto mb-lg-0">
              <li className="nav-item">
                <Link className={`nav-link ${pathname === `/${locale}` || pathname === `/${locale}/` ? 'active' : ''}`} href={`/${locale}`}>
                  {t("header.home")}
                </Link>
              </li>
              <li className="nav-item">
                <Link className={`nav-link ${pathname.includes('/how-it-works') ? 'active' : ''}`} href={`/${locale}/how-it-works`}>
                  {t("header.how_it_works")}
                </Link>
              </li>
              <li className="nav-item">
                <Link className={`nav-link ${pathname.includes('/projects') && !pathname.includes('/my-projects') && !pathname.includes('/post-project') ? 'active' : ''}`} href={`/${locale}/projects`}>
                  {t("header.find_projects")}
                </Link>
              </li>
              <li className="nav-item">
                <Link className={`nav-link ${pathname.includes('/categories') ? 'active' : ''}`} href={`/${locale}/categories`}>
                  {t("header.service_category")}
                </Link>
              </li>
              <li className="nav-item">
                <Link className={`nav-link ${pathname.includes('/freelancers') && !pathname.includes('/freelancer/') ? 'active' : ''}`} href={`/${locale}/freelancers`}>
                  {t("header.freelancers")}
                </Link>
              </li>
            </ul>
            <div className="d-flex button-gropup align-items-center login-signup-nav">
              {status === 'authenticated' && session?.user ? (
                <>
                  {isClient && (
                    <div className="post-project-link">
                      <Link 
                        href={`/${locale}/client/post-project`} 
                        className="custom-btn primary-outline text-uppercase"
                      >
                        {t("header.post_project")}
                      </Link>
                    </div>
                  )}
                </>
              ) : (
                <>
                  <div className="post-project-link">
                    <Link 
                      href={`/${locale}/client/post-project`} 
                      className="custom-btn primary-outline text-uppercase"
                    >
                      {t("header.post_project")}
                    </Link>
                  </div>
                  <Link 
                    href={`/${locale}/login`} 
                    className={`nav-btn ${pathname.includes('/login') ? 'active' : ''}`}
                  >
                    {t("header.login")}
                  </Link>
                  <Link 
                    href={`/${locale}/signup`} 
                    className={`nav-btn ${pathname.includes('/signup') ? 'active' : ''}`}
                  >
                    {t("header.signup")}
                  </Link>
                </>
              )}
            </div>
          </div>

          <div className="d-flex align-items-center gap-2">
            {status === 'authenticated' && session?.user && (
              <div className="d-flex button-gropup loginuser-nav ps-4">
                {/* Notifications */}
                <NotificationPanel />

                {/* Messages Dropdown */}
                <MessagesDropdown />

                {/* User Profile Dropdown */}
                <div className="language-selector userlogin-dropdown">
                  <div className="nav-item dropdown language-dropdown">
                    <button type="button" className="nav-link select-lang">
                      <Image 
                        src="/assets/image/loginuser.png" 
                        alt="user" 
                        width={0}
                        height={0}
                        sizes="100vw"
                        style={{ width: '40px', height: '40px' }}
                        className="me-2"
                      />
                      <div>
                        {session.user?.name || 'User'}
                      </div>
                    </button>
                    <ul>
                      <li>
                        <Link 
                          className={pathname.includes('/profile') ? 'active' : ''} 
                          href={`/${locale}/${userRole.toLowerCase()}/profile`}
                        >
                          <Image 
                            src="/assets/image/profile-circle.svg" 
                            alt="My Profile" 
                            width={0}
                            height={0}
                            sizes="100vw"
                            style={{ width: '16px', height: '16px' }}
                            className="me-2"
                          />
                          {t("header.profile")}
                        </Link>
                      </li>
                      <li>
                        <Link 
                          className={pathname.includes('/message') ? 'active' : ''} 
                          href={`/${locale}/${userRole.toLowerCase()}/messages`}
                        >
                          <Image 
                            src="/assets/image/user-edit.svg" 
                            alt="Messages" 
                            width={0}
                            height={0}
                            sizes="100vw"
                            style={{ width: '16px', height: '16px' }}
                            className="me-2"
                          />
                          {t("header.message")}
                        </Link>
                      </li>
                      <li>
                        <Link 
                          className={pathname.includes(isClient ? '/my-projects' : '/my-projects') ? 'active' : ''} 
                          href={isClient ? `/${locale}/client/my-projects` : `/${locale}/freelancer/my-projects`}
                        >
                          <Image 
                            src="/assets/image/my-project.svg" 
                            alt="My Projects" 
                            width={0}
                            height={0}
                            sizes="100vw"
                            style={{ width: '16px', height: '16px' }}
                            className="me-2"
                          />
                          {isClient ? 'My Projects' : 'My Proposals'}
                        </Link>
                      </li>
                      <li>
                        <button 
                          type="button" 
                          onClick={handleSignOut} 
                          className="dropdown-item"
                          style={{paddingLeft: '15px'}}
                        >
                          <Image 
                            src="/assets/image/logout.svg" 
                            alt="Logout" 
                            width={0}
                            height={0}
                            sizes="100vw"
                            style={{ width: '16px', height: '16px' }}
                            className="me-2"
                          />
                          {t("header.logout")}
                        </button>
                      </li>
                    </ul>
                  </div>
                </div>


              </div>
            )}
                <button 
                  className="navbar-toggler" 
                  type="button" 
                  data-bs-toggle="collapse" 
                  data-bs-target="#navbarSupportedContent" 
                  aria-controls="navbarSupportedContent" 
                  aria-expanded="false" 
                  aria-label="Toggle navigation"
                >
                  <span className="navbar-toggler-icon"></span>
                </button>
          </div>
        </div>
      </nav>
    </header>
  );
}
