import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/auth";
import { prisma } from "@/lib/prisma";
import bcrypt from "bcryptjs";

export async function GET(req: NextRequest) {
  try {
    // Check if required dependencies are available
    if (!prisma) {
      return NextResponse.json(
        { error: "Database connection not available" },
        { status: 503 }
      );
    }
    const session = await getServerSession(authOptions);

    if (!session?.user?.email) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    const userId = session?.user?.id; // or any user ID you have

    const conversations = await prisma.conversation.findMany({
      where: {
        OR: [{ buyerId: userId }, { sellerId: userId }],
      },
      distinct: ["projectId"],
      include: {
        // project: true,      // optional: include related project details
        buyer: {
          select: {
            id: true,
            email: true,
            profile: {
              select: {
                firstName: true,
                lastName: true,
                profileImage: true,
              },
            },
          },
        },
        seller: {
          select: {
            id: true,
            email: true,
            profile: {
              select: {
                firstName: true,
                lastName: true,
                profileImage: true,
              },
            },
          },
        },
        // chats: {
        //   take: 1,
        //   orderBy: { createdAt: 'desc' }, // get latest message
        // },
      },
      orderBy: {
        updatedAt: "desc", // show most recent conversations first
      },
    });

    // Then get conversations grouped by project
    const conversationsByProject = await Promise.all(
      conversations.map(async (userProject) => {
        const conversations = await prisma.conversation.findMany({
          where: {
            projectId: userProject.projectId,
            OR: [{ buyerId: userId }, { sellerId: userId }],
          },
          include: {
            buyer: {
              select: {
                id: true,
                email: true,
                profile: {
                  select: {
                    firstName: true,
                    lastName: true,
                    profileImage: true,
                  },
                },
              },
            },
            seller: {
              select: {
                id: true,
                email: true,
                profile: {
                  select: {
                    firstName: true,
                    lastName: true,
                    profileImage: true,
                  },
                },
              },
            },
            // chats: {
            //   take: 1,
            //   orderBy: { createdAt: 'desc' },
            // },
          },
          orderBy: {
            updatedAt: "desc",
          },
        });

        return {
          projectId: userProject.projectId,
          project: userProject.productSnapshot,
          conversations: conversations,
        };
      })
    );

    return NextResponse.json({
      message: "Conversation created successfully",
      // conversationData: conversations,
      conversationData: conversationsByProject,
    });
  } catch (dbError) {
    console.error("Database error during password change:", dbError);
    throw dbError;
  }
}
