'use client';

import { useState, useRef } from 'react';

interface MessageInputProps {
  onSendMessage: (content: string, attachments?: any[]) => Promise<void>;
  onTyping?: (isTyping: boolean) => void;
  disabled?: boolean;
  placeholder?: string;
}

export default function MessageInput({ onSendMessage, onTyping, disabled, placeholder = "Type your message..." }: MessageInputProps) {
  const [message, setMessage] = useState('');
  const [sending, setSending] = useState(false);
  const [uploading, setUploading] = useState(false);
  const [attachments, setAttachments] = useState<any[]>([]);
  const [isTyping, setIsTyping] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const typingTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  const handleSubmit = async () => {
    if (!message.trim() || sending || disabled) {
      return;
    }

    const messageContent = message.trim();
    setMessage('');
    setSending(true);

    // Stop typing indicator when sending
    if (isTyping) {
      setIsTyping(false);
      onTyping?.(false);
      if (typingTimeoutRef.current) {
        clearTimeout(typingTimeoutRef.current);
      }
    }

    try {
      await onSendMessage(messageContent, attachments);
      setAttachments([]);
    } catch (error) {
      // Restore message on error
      setMessage(messageContent);
    } finally {
      setSending(false);
    }
  };

  const handleTypingChange = (value: string) => {
    setMessage(value);
    
    if (onTyping) {
      // Start typing if not already typing and there's content
      if (!isTyping && value.trim()) {
        setIsTyping(true);
        onTyping(true);
      }

      // Clear existing timeout
      if (typingTimeoutRef.current) {
        clearTimeout(typingTimeoutRef.current);
      }

      // Set timeout to stop typing after 2 seconds of inactivity
      if (value.trim()) {
        typingTimeoutRef.current = setTimeout(() => {
          setIsTyping(false);
          onTyping(false);
        }, 2000);
      } else if (isTyping) {
        // Stop typing immediately if message is empty
        setIsTyping(false);
        onTyping(false);
      }
    }
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSubmit();
    }
  };

  const handleAttachClick = () => {
    fileInputRef.current?.click();
  };

  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;
    setUploading(true);
    try {
      const form = new FormData();
      form.append('file', file);
      const res = await fetch('/api/messages/upload', {
        method: 'POST',
        body: form,
      });
      if (!res.ok) throw new Error('Upload failed');
      const data = await res.json();
      setAttachments((prev) => [...prev, { url: data.url, name: data.name, type: data.type }]);
    } catch (err) {
    } finally {
      setUploading(false);
      if (fileInputRef.current) fileInputRef.current.value = '';
    }
  };

  return (
    <div className="message-input">
      <div className="wrap">
        <input 
          type="text" 
          placeholder={placeholder}
          value={message}
          onChange={(e) => handleTypingChange(e.target.value)}
          onKeyDown={handleKeyDown}
          disabled={disabled || sending}
        />
        <div className="chat-file-upload">
          <input 
            ref={fileInputRef}
            type="file" 
            name="fileupload" 
            id="fileupload"
            onChange={handleFileChange}
          />
          <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M13.7298 5.73091C14.199 5.26161 14.8354 4.9979 15.499 4.99781C16.1626 4.99772 16.799 5.26124 17.2683 5.73041C17.7376 6.19957 18.0013 6.83596 18.0014 7.49955C18.0015 8.16315 17.738 8.79961 17.2688 9.26891L10.7088 15.7089L10.7058 15.7119C10.5172 15.8941 10.2646 15.9949 10.0024 15.9926C9.74022 15.9903 9.48941 15.8851 9.304 15.6997C9.11859 15.5143 9.01342 15.2635 9.01114 15.0013C9.00887 14.7391 9.10966 14.4865 9.29182 14.2979L15.7318 7.85791C15.8229 7.76361 15.8733 7.6373 15.8722 7.50621C15.871 7.37511 15.8184 7.2497 15.7257 7.157C15.633 7.06429 15.5076 7.01171 15.3765 7.01057C15.2454 7.00943 15.1191 7.05983 15.0248 7.15091L8.58482 13.5909C8.2107 13.9649 7.99999 14.4719 7.99882 15.0009C7.99882 16.1009 8.89382 17.0009 9.99882 17.0009C10.5275 17.0003 11.0345 16.7903 11.4088 16.4169L17.9688 9.97691L17.9718 9.97391C18.6288 9.31952 18.9992 8.43117 19.0018 7.50391C19.0018 5.57391 17.4318 4.00391 15.5018 4.00391C14.576 4.00384 13.6878 4.3706 13.0318 5.02391L6.47182 11.4639L6.46882 11.4669C6.00391 11.9314 5.63555 12.4834 5.38497 13.091C5.13439 13.6985 5.00655 14.3497 5.00882 15.0069C5.00882 17.7669 7.24882 20.0069 10.0088 20.0069C11.3888 20.0069 12.6388 19.4469 13.5488 18.5469L17.8688 14.2269C17.9599 14.1326 18.0103 14.0063 18.0092 13.8752C18.008 13.7441 17.9554 13.6187 17.8627 13.526C17.77 13.4333 17.6446 13.3807 17.5135 13.3796C17.3824 13.3784 17.2561 13.4288 17.1618 13.5199L12.8418 17.8399C12.4707 18.2121 12.0296 18.5072 11.5439 18.708C11.0581 18.9088 10.5374 19.0114 10.0118 19.0099C7.80182 19.0099 6.01182 17.2199 6.01182 15.0099C6.01182 13.9099 6.45882 12.9099 7.18182 12.1799L13.7418 5.73991L13.7448 5.73691L13.7298 5.73091Z" fill="#564B43"/>
          </svg>
        </div>
        <button 
          className="button-primary"
          onClick={handleSubmit}
          disabled={disabled || sending || !message.trim()}
        >
          {sending ? (
            <div className="spinner-border spinner-border-sm" role="status">
              <span className="visually-hidden">Sending...</span>
            </div>
          ) : (
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M6.50862 20.125C5.57528 20.125 4.98362 19.8083 4.60862 19.4333C3.87528 18.7 3.35862 17.1416 5.00862 13.8333L5.73362 12.3916C5.82528 12.2 5.82528 11.8 5.73362 11.6083L5.00862 10.1666C3.35028 6.85829 3.87528 5.29162 4.60862 4.56662C5.33362 3.83329 6.90028 3.30829 10.2003 4.96662L17.3336 8.53329C19.1086 9.41662 20.0836 10.65 20.0836 12C20.0836 13.35 19.1086 14.5833 17.3419 15.4666L10.2086 19.0333C8.59195 19.8416 7.39195 20.125 6.50862 20.125ZM6.50862 5.12496C6.05862 5.12496 5.70862 5.23329 5.49195 5.44996C4.88362 6.04996 5.12528 7.60829 6.12528 9.59996L6.85028 11.05C7.11695 11.5916 7.11695 12.4083 6.85028 12.95L6.12528 14.3916C5.12528 16.3916 4.88362 17.9416 5.49195 18.5416C6.09195 19.15 7.65028 18.9083 9.65028 17.9083L16.7836 14.3416C18.0919 13.6916 18.8336 12.8333 18.8336 11.9916C18.8336 11.15 18.0836 10.2916 16.7753 9.64162L9.64195 6.08329C8.37528 5.44996 7.28362 5.12496 6.50862 5.12496Z" fill="#564B43"/>
              <path d="M11.0332 12.625H6.5332C6.19154 12.625 5.9082 12.3417 5.9082 12C5.9082 11.6583 6.19154 11.375 6.5332 11.375H11.0332C11.3749 11.375 11.6582 11.6583 11.6582 12C11.6582 12.3417 11.3749 12.625 11.0332 12.625Z" fill="#564B43"/>
            </svg>
          )}
        </button>
      </div>
      {attachments.length > 0 && (
        <div className="mt-2 small text-muted">
          {attachments.map((a, i) => (
            <span key={i} className="me-2">📎 {a.name}</span>
          ))}
        </div>
      )}
    </div>
  );
}