<?php

namespace App\Http\Controllers\Admin;

use App\DataTables\ProductDataTable;
use App\Exports\ProductExport;
use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Inventory;
use App\Models\Product;
use App\Models\UnitType;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\View;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;

class ProductController extends Controller
{
    public function index(ProductDataTable $dataTable)
    {
        abort_if(Gate::denies('product_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        return $dataTable->render('admin.product.index');
    }

    public function create(Request $request)
    {
        abort_if(Gate::denies('product_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $product_units = UnitType::all();
        $brands = Brand::all();
        if($request->ajax()){
            $htmlView = view('admin.product.create', compact('brands', 'product_units'))->render();
           return response()->json(['success' => true, 'htmlView' => $htmlView]);
        }
    }

    public function store(Request $request)
    {
        abort_if(Gate::denies('product_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $rules = [
            'name' => 'required|string|max:250|unique:products,name,NULL,id,brand_id,' . $request->brand_id . ',unit_type,' . $request->unit_type,
            'brand_id' => 'required|numeric|exists:brands,id',
            'unit_type' => 'required|numeric|exists:unit_types,id',
            'calculation' => 'required|numeric'
        ];
        $validator = Validator::make($request->all(), $rules);
         if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->toArray()]);
        }

        $data = [
            'name' => $request->name,
            'brand_id' => $request->brand_id,
            'unit_type' => $request->unit_type,
            'calculation' => $request->calculation,            
        ];       
        $product =  Product::create($data);
        addToLog($request, 'Product', 'Create', $product);
        $data = Inventory::create([
            'product_id' => $product->id, 
            'entry_date' => Carbon::today()->toDateString(),
            'quantity'   => 0,
        ]);
        addToLog($request,'Inventory','Create', $data);
        return response()->json(['success' => 'Product Created Successfully.', 'product' => $product]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, string $id)
    {
        abort_if(Gate::denies('product_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        if ($request->ajax()) {
            $id = decrypt($id);
            $product = Product::where('id', $id)->first();
            $html = View::make('admin.product.show', compact('product'))->render();
            return response()->json(['success' => true, 'html' => $html]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        abort_if(Gate::denies('product_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $product = Product::findOrFail($id);
        $product_units = UnitType::all();
        $brands = Brand::all();   
        $htmlView = view('admin.product.edit', compact('brands', 'product_units','product'))->render();
        return response()->json(['success' => true, 'htmlView' => $htmlView]);
        
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        abort_if(Gate::denies('product_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $rules = [
            'name' => 'required|string|max:250|unique:products,name,' . $id . ',id,brand_id,' . $request->brand_id . ',unit_type,' . $request->unit_type,
            'brand_id' => 'required|numeric',
            'unit_type' => 'required|numeric',
            'calculation' => 'required|numeric'
        ];
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'error' => $validator->errors()->toArray()
            ]);
        }
        $product = Product::findOrFail($id);
        $oldvalue = $product->getOriginal();
        $product->name = $request->name;
        $product->brand_id = $request->brand_id;
        $product->unit_type = $request->unit_type;
        $product->calculation = $request->calculation;       
        $product->save();
        $newValue = $product->refresh();
        addToLog($request, 'Product', 'Edit', $newValue, $oldvalue);
        return response()->json(['success' => 'Product Updated successfully.']);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id)
    {
        abort_if(Gate::denies('product_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $record = Product::find(decrypt($id));
        $oldvalue = $record->getOriginal();
        $record->updated_by = Auth::id();
        $record->save();
        $newValue = $record->refresh();
        addToLog($request, 'Product', 'Delete', $newValue, $oldvalue);
        $record->delete();
        return response()->json(['success' => 'Product Delete successfully.']);
    }

   

    public function export($product_id = null)
    {
        abort_if(Gate::denies('product_export'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        return Excel::download(new ProductExport($product_id), 'product-list.xlsx');
    } 
    
    
    

    public function getProductToSearch(Request $request){
        // dd("serch id");
        // if($request->ajax()){
        //     $brand_id = $request->brand_id;


        //    // $category = ProductCategory::where('brand_id',$brand_id)->pluck('name', 'id')->prepend(trans('admin_master.g_please_select'), '');
        //     $html = View::make('admin.master.category.category_list',compact('category','selected_id'))->render();
        //     return response()->json(['success' => true, 'html' => $html]);
        // }
    }

    public function allProducts()
    {
        $allproduct = Product::orderBy('name')->get();        
        return response()->json(['success' => true, 'allproduct' => $allproduct]);
    }
}
