<?php

namespace App\Http\Controllers\Admin;

use App\DataTables\InvoiceDataTable;
use App\Http\Controllers\Controller;
use App\Http\Requests\Order\StoreRequest;
use App\Http\Requests\Order\UpdateRequest;
use App\Models\Customer;
use App\Models\Inventory;
use App\Models\Order;
use App\Models\OrderProduct;
use App\Models\PaymentTransaction;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use App\Models\ShiftingGood;
use Illuminate\Support\Facades\Gate;
use App\Models\CashManagement;
use Illuminate\Support\Facades\Auth;

class OrderController extends Controller
{
    public function index(InvoiceDataTable $dataTable)
    {
        abort_if(Gate::denies('estimate_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $customers = Customer::all();
        $type='all';
        return $dataTable->render('admin.order.index',compact('customers','type'));
    }

    public function create()
    {
        abort_if(Gate::denies('estimate_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $allproduct = Product::orderBy('name')->get();        
        $allcustomer = Customer::orderBy('name')->get();
        return view('admin.order.create',compact('allproduct','allcustomer'));
    }   

    public function validateStockAvailability(Request $request)
    {
        $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'quantity' => 'required|numeric',
            'dispatch_place' => 'required|string|in:warehouse,shop,cold_storage',
        ]);        

        $stockAvailable = $this->checkStockAvailability($request->product_id,$request->dispatch_place);
        if($stockAvailable  >= $request->quantity){
            return response()->json(['available' => true , 'available_stock'=>$stockAvailable]);
        }else{
            return response()->json(['available' => false , 'available_stock'=>$stockAvailable]);
        }
    }  

    public function getStockAvailability(Request $request)
    {
        $request->validate([
            'product_id' => 'required|integer|exists:products,id',            
        ]);            
        $inventory = [];

        $stockBuyData = Inventory::where('product_id', $request->product_id)
        ->select('place', DB::raw('SUM(quantity) as stock_buy_count'))
        ->groupBy('place')->pluck('stock_buy_count', 'place');

        $stockSellData = OrderProduct::where('product_id', $request->product_id)
            ->select('dispatch_place', DB::raw('SUM(quantity) as stock_sell_count'))
            ->groupBy('dispatch_place')->pluck('stock_sell_count', 'dispatch_place');        

        $shiftedFromData = ShiftingGood::where('product_id', $request->product_id)
            ->select('from', DB::raw('SUM(quantity) as shifted_count'))
            ->groupBy('from')
            ->pluck('shifted_count', 'from');

        $places = array_keys(config('app.product_place'));

        // Map the available stock at each place (include cases with 0 stock)
        $availableStock = collect($places)->map(function ($place) use ($stockBuyData, $stockSellData, $shiftedFromData, &$inventory) {
            $buyCount = $stockBuyData[$place] ?? 0;  // Default to 0 if no stock was bought at this place
            $sellCount = $stockSellData[$place] ?? 0;  // Default to 0 if no stock was sold at this place
            $shiftCount = $shiftedFromData[$place] ?? 0; // Shifted from this place
            $inventory[$place] = ($buyCount - $sellCount - $shiftCount);
            return config('app.product_place.'.$place) . ' : ' . ($buyCount - $sellCount - $shiftCount);
        })->implode(', ');
        
        $availableStock = rtrim($availableStock, ' , ');   

        return response()->json([
            'success' => true,
            'message' => $availableStock,
            'inventory' => $inventory
        ], 200);
    }   

    public function store(StoreRequest $request)
    {        
        // Check Credit Limit of the Customer
        $customer = Customer::where('id', $request->customer_id)->first();
        $creditLimitWarning = null;
        $isCashOrPhonePay = in_array($customer->customer_type, ['cash', 'phonepay']);
        if($customer->credit_limit > 0){
            $balance = $customer->credit_limit - ($customer->debit_balance - $customer->credit_balance);

            if($balance  < $request->grand_total){
                /*return response()->json([
                    'success' => false,
                    'message' => 'Customer Credit Limit is '. $customer->credit_limit ,
                    'alert-type'=> trans('quickadmin.alert-type.error'),
                    'title'     => 'Credit Limit Exceeded !',                       
                ], 400);*/
                
                $creditLimitWarning = [
                    'message' => 'Customer Credit Limit is '. $customer->credit_limit ,
                    'alertType'=> trans('quickadmin.alert-type.warning'),
                    'title'     => 'Credit Limit Exceeded !',
                ];
            }
        }       

        // Consolidate products by product_id and sum their quantities
        $productQuantities = [];
        foreach ($request->products as $productData) {
            $productKey = $productData['product_id'] . '-' . $productData['dispatch_place'];
            if (!isset($productQuantities[$productKey])) {
                $productQuantities[$productKey] = 0;
            }
            $productQuantities[$productKey] += $productData['quantity'];
        }

        // Check stock availability for each consolidated product entry
        $stockAvailabilityMessages = [];
        foreach ($productQuantities as $productKey => $totalQuantity) {
            list($productId, $dispatchPlace) = explode('-', $productKey);            
            $availableStock = $this->checkStockAvailability($productId, $dispatchPlace);
            if ($availableStock < $totalQuantity) {
                $product = Product::find($productId);
                $stockAvailabilityMessages[] = $product->name . ' at ' . ucfirst($dispatchPlace) . ' has only ' . $availableStock . ' stock available.';
            }
        }


        // if (!empty($stockAvailabilityMessages)) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => implode(' , ', $stockAvailabilityMessages),
        //         'alert-type' => trans('quickadmin.alert-type.error'),
        //         'title'     => 'Insufficient Stock',
        //     ], 400);
        // }        

        try
        { 
            DB::beginTransaction();
            // Create a new order        
            $order = Order::create([
                'customer_id'   => $request->customer_id,
                'labour_bhada'  => $request->labour_bhada,
                'gadi_bhada'    => $request->gadi_bhada,
                // 'is_round_off' => $request->is_round_off,
                'sub_total'     => $request->sub_total,
                // 'round_off_amount' => $request->round_off_amount,
                'grand_total'   => $request->grand_total,
                'invoice_date'  => $request->invoice_date,
                'remark'        => $request->remark,
            ]);
            $invoiceNumber = getNewInvoiceNumber('', 'new',$request->invoice_date);         
            $order->update(['invoice_number' => $invoiceNumber]);
            $transaction = [
                'order_id'          => $order->id,
                'customer_id'       => $order->customer_id,
                'payment_type'      => 'debit',
                'payment_way'       => 'order_create',
                'generated_from'    => 'Cash',
                'voucher_number'    => $order->invoice_number,
                'amount'            => $request->grand_total,                
                'entry_date'        => $request->invoice_date,
                'remark'            => 'Sales',
            ];
            $createdTransaction = PaymentTransaction::create($transaction);
            
            if($isCashOrPhonePay){
                $paymentWay = ($isCashOrPhonePay) ? 'by_' . $customer->customer_type : 'order_create';
                $transaction = [
                    'order_id'          => $order->id,
                    'customer_id'       => $order->customer_id,
                    'payment_type'      => 'credit',
                    'payment_way'       => 'by_cash',
                    'generated_from'    => 'Order',
                    'voucher_number'    => null,
                    'amount'            => $request->grand_total,                
                    'entry_date'        => $request->invoice_date,
                    'remark'            => 'Cash receipt generated by ' . Auth::user()->name . ' on ' . now()->format('d-m-Y'),
                ];
                $createdTransaction = PaymentTransaction::create($transaction);

                $createdTransaction->cashReceiptAmounts()->create([
                    'payment_transaction_id' => $createdTransaction->id,
                    'amount'                 => $createdTransaction->amount,
                    'payment_way'            => $customer->customer_type == 'cash' ? 'by_cash' : 'by_phonepay'
                ]);
                if($customer->customer_type == 'cash'){
                    storeCashBalanceRecord($createdTransaction, 'credit',$request->invoice_date);
                }
            }

            // Create order products           
            foreach ($request->products as $productData) {
                OrderProduct::create([
                    'order_id'          => $order->id,
                    'product_id'        => $productData['product_id'],
                    'quantity'          => $productData['quantity'],
                    'price'             => $productData['price'],
                    'total_price'       => $productData['total_price'],
                    'dispatch_place'    => $productData['dispatch_place'],
                ]);
            }

            addToLog($request, 'Estimate', 'Create', $transaction);

            DB::commit();

            return response()->json([
                'success' => true,
                'creditLimitWarning' => $creditLimitWarning,
                'message' => trans('messages.crud.add_record'),
                'alert-type'=> trans('quickadmin.alert-type.success'),
                'invoiceNumber'     => 'Invoice No. '.$order->invoice_number,
                'redirectUrl' => route('orders.index'),
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
        //   dd($e->getMessage());
            return response()->json(['success' => false,
            'message' => trans('messages.error1'),
            'alert-type'=> trans('quickadmin.alert-type.error')], 500);
        }
    }

    public function edit(Order $order)
    {
        $order->load('orderProduct.product');
        $customer_id = $order->customer->id;
        $products = $order->orderProduct->map(function ($product) use ($customer_id) {
        $associatedProduct = Product::find($product->product_id);
            return [
                'order_product_id'  =>  $product->id,                
                'product_id'        => $product->product_id,
                'product_name'      => $associatedProduct->full_name,
                'brand_name'        => $associatedProduct->brands->name,
                'unit_name'         => $associatedProduct->product_unit->name,
                'quantity'          => $product->quantity,
                'price'             => $product->price,
                'total_price'       => $product->total_price,
                'dispatch_place'    => $product->dispatch_place,                
            ];
        })->toArray();

        $orderData = [
            'order_id'          => $order->id,
            'invoice_number'    => $order->invoice_number,
            'customer_id'       => $customer_id,
            'products'          => $products,
            'labour_bhada'      => $order->labour_bhada,
            'gadi_bhada'        => $order->gadi_bhada,
            // 'is_round_off'   => $order->is_round_off,
            'sub_total'         => $order->sub_total,
            // 'round_off_amount' => $order->round_off,
            'grand_total'       => $order->grand_total,
            'invoice_date'      => $order->invoice_date,
            'remark'            => $order->remark,
        ];

        $invoice_date = $order->invoice_date ? $order->invoice_date->format('d-m-Y') : null;        
        $allproduct = Product::orderBy('name')->get();        
        $allcustomer = Customer::orderBy('name')->get();        
        
        return view('admin.order.edit',compact('order','invoice_date','allcustomer','allproduct','orderData'));
    }

    public function update(UpdateRequest $request, Order $order)
    {
        // Check Credit Limit of the Customer
        $customer = Customer::where('id', $request->customer_id)->first();            
        $creditLimitWarning = null;
        $isCashOrPhonePay = in_array($customer->customer_type, ['cash', 'phonepay']);
        if($customer->credit_limit > 0){
            $debitBalnce = $customer->transaction()
                ->where('payment_type', 'debit')
                ->where('voucher_number','!=',$order->invoice_number)
                ->sum('amount');            

            $balance = $customer->credit_limit - ($debitBalnce - $customer->credit_balance);

            if($balance <= $request->grand_total){
                /*return response()->json([
                    'success' => false,
                    'message' => 'Customer Credit Limit is '. $customer->credit_limit ,
                    'alert-type'=> trans('quickadmin.alert-type.error'),
                    'title'     => 'Credit Limit Exceeded !',                       
                ], 500);*/
                $creditLimitWarning = [
                    'message' => 'Customer Credit Limit is '. $customer->credit_limit ,
                    'alertType'=> trans('quickadmin.alert-type.warning'),
                    'title'     => 'Credit Limit Exceeded !',
                ];
            }
        }     

        try {
            DB::beginTransaction();            

            // Update order
            $order->update([
                'customer_id'   => $request->customer_id,
                'labour_bhada'  => $request->labour_bhada,
                'gadi_bhada'    => $request->gadi_bhada,
                // 'is_round_off'   => $request->is_round_off,
                'sub_total'     => $request->sub_total,
                // 'round_off_amount' => $request->round_off_amount,
                'grand_total'   => $request->grand_total,
                'invoice_date'  => $request->invoice_date,
                'remark'        => 'Cash receipt generated by ' . Auth::user()->name . ' [' . now()->format('d-m-Y H:i') . ']',
            ]);

            // NEW TRANSACTION LOGIC LIKE STORE
            $invoiceNumber = $order->invoice_number; 
            $transaction = [
                'order_id'        => $order->id,
                'customer_id'     => $order->customer_id,
                'payment_type'    => 'debit',
                'payment_way'     => 'order_create',
                'generated_from'  => 'Cash',
                'voucher_number'  => $order->invoice_number,
                'amount'          => $request->grand_total,
                'entry_date'      => $request->invoice_date,
                'remark'          => 'Sales',
            ];
            $createdTransaction = PaymentTransaction::updateOrCreate(
                ['order_id' => $order->id, 'voucher_number' => $order->invoice_number],
                $transaction
            );

            if($isCashOrPhonePay){
                $paymentWay = 'by_' . $customer->customer_type;
                $transaction = [
                    'order_id'        => $order->id,
                    'customer_id'     => $order->customer_id,
                    'payment_type'    => $isCashOrPhonePay ? 'credit' : 'debit',
                    'payment_way'     => $paymentWay,
                    'generated_from'  => $isCashOrPhonePay ? 'Order' : 'Cash',
                    'voucher_number'  => !$isCashOrPhonePay ? $order->invoice_number : null,
                    'amount'          => $request->grand_total,
                    'entry_date'      => $request->invoice_date,
                    'remark'          => 'Sales',
                ];
                $createdTransaction = PaymentTransaction::updateOrCreate(
                    ['order_id' => $order->id, 'payment_type' => $transaction['payment_type']],
                    $transaction
                );

                if($customer->customer_type == 'cash'){
                    storeCashBalanceRecord($createdTransaction, 'credit', $request->invoice_date);
                }
            }
            // END TRANSACTION LOGIC
            $productsToDelete = $request->deleted_products ?? [];        

            foreach ($request->products as $productData) {
                if (isset($productData['order_product_id']) && !empty($productData['order_product_id'])) {
                    // Update existing product with order_product_id
                    $existingProduct = OrderProduct::find($productData['order_product_id']);
                    if ($existingProduct) {
                        $existingProduct->update([
                            'quantity'        => $productData['quantity'],
                            'product_id'      => $productData['product_id'],
                            'price'           => $productData['price'],
                            'total_price'     => $productData['total_price'],
                            'dispatch_place'  => $productData['dispatch_place'],                            
                        ]);
                    }
                } else {
                    // Create a new product without order_product_id
                    OrderProduct::create([
                        'order_id'       => $order->id,
                        'product_id'     => $productData['product_id'],
                        'quantity'       => $productData['quantity'],
                        'price'          => $productData['price'],
                        'total_price'    => $productData['total_price'],
                        'dispatch_place' => $productData['dispatch_place'],
                    ]);
                }
            }

            if (!empty($productsToDelete)) {
                OrderProduct::whereIn('id', $productsToDelete)->delete();
            }

            addToLog($request, 'Estimate', 'Edit', $transaction);

            DB::commit();

            return response()->json([                
                'invoiceNumber'       => 'Invoice No. '.$order->invoice_number,
                'redirectUrl'         => route('orders.index'),
                'success'             => true,
                'creditLimitWarning'  => $creditLimitWarning,
                'message'             => trans('messages.crud.update_record'),
                'alert-type'          => trans('quickadmin.alert-type.success')
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => trans('messages.error1'),
                'alert-type' => trans('quickadmin.alert-type.error')
            ], 500);
        }
    }


    public function destroy(Order $order)
    {
        abort_if(Gate::denies('estimate_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        try{
            DB::beginTransaction();
            $transaction =  PaymentTransaction::where('voucher_number',$order->invoice_number)->first(); 
            if (!$transaction) {
                $transaction = PaymentTransaction::where('order_id', $order->id)->first();
            }
            if ($transaction) {
                $transaction->delete();
            }
            // $transaction->delete();
            $order->delete();
            Db::commit();
            return response()->json(['success' => true,
            'message' => trans('messages.crud.delete_record'),
            'alert-type'=> trans('quickadmin.alert-type.success'),
            'title' => trans('quickadmin.order.invoice')
            ], 200);

        }catch(\Exception $e){
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => trans('messages.error1'),
                'alert-type' => trans('quickadmin.alert-type.error')
            ], 500);
        }             
    }

    public function show(Order $order)
    {        
        abort_if(Gate::denies('inventory_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');        
        $htmlView = view('admin.order.show',compact('order'))->render();
        return response()->json(['success' => true, 'htmlView' => $htmlView]);
    }   

    public function printOrder(Order $order){        
        abort_if(Gate::denies('estimate_print'), Response::HTTP_FORBIDDEN, '403 Forbidden');     
        return view('admin.order._print_order',compact('order'))->render(); 
    }


    public function checkStockAvailability($product_id , $place)
    {          
        $stockBuyAvailable =  Inventory::where('product_id', $product_id)
        ->where('place', $place)       
        ->sum('quantity');
                
        $stockSellAvailable =  OrderProduct::where('product_id',$product_id)
        ->where('dispatch_place', $place)       
        ->sum('quantity');
        
        $availableStock = $stockBuyAvailable - $stockSellAvailable ;
        return $availableStock;
    }
}
