<?php

namespace App\Http\Controllers\Admin;

use App\DataTables\InventoryDataTable;
use App\DataTables\InventoryViewDataTable;
use App\Http\Controllers\Controller;
use App\Http\Requests\Inventory\StoreRequest;
use App\Http\Requests\Inventory\UpdateRequest;
use App\Http\Requests\ShiftingGood\StoreRequest as ShiftingGoodRequest;
use App\Models\Area;
use App\Models\Inventory;
use App\Models\Product;
use App\Models\OrderProduct;
use App\Models\ShiftingGood;
use App\Models\Brand;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Exception;
class InventoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    protected $storageTypes;

    public function __construct()
    {
        $this->storageTypes = config('app.product_place');
    }

    public function index(InventoryDataTable $dataTable)
    {
        $allproduct = Product::orderBy('name')->get();        
        abort_if(Gate::denies('inventory_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        return $dataTable->render('admin.inventory.index', compact('allproduct'));
    }

    /**
     * Show the form for creating a new resource.
     */

    public function show(Inventory $inventory)
    {      
       abort_if(Gate::denies('inventory_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
       $htmlView = view('admin.inventory.view',compact('inventory'))->render();
       return response()->json(['success' => true, 'htmlView' => $htmlView]);
    } 

    public function shiftingGoodShow($id)
    {   
        try {
            $shiftingGood = ShiftingGood::find($id);
            $htmlView = view('admin.inventory.view',compact('shiftingGood'))->render();
            return response()->json(['success' => true, 'htmlView' => $htmlView]);
        } catch (Exception $e) {
              return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'alert-type' => trans('quickadmin.alert-type.error'),
                'title' => trans('quickadmin.inventory.shifting_goods'),
            ], 500);
        }
    } 

    public function create()
    {
        abort_if(Gate::denies('inventory_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $allproduct = Product::orderBy('name')->get();        
        $htmlView = view('admin.inventory.create',compact('allproduct'))->render();
        return response()->json(['success' => true, 'htmlView' => $htmlView]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreRequest $request)
    {
        $data = $request->all();  
        $data['type'] = 'add';
        $inventoryData = Inventory::create($data);
        addToLog($request,'Inventory','Create', $inventoryData);
        return response()->json(['success' => true,
        'message' => trans('messages.crud.add_record'),
        'alert-type'=> trans('quickadmin.alert-type.success'),
        'title' => trans('quickadmin.inventory.inventory'),
        ], 200);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Inventory $inventory)
    {        
        abort_if(Gate::denies('inventory_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $allproduct = Product::orderBy('name')->get();
        $allarea =  Area::orderBy('address')->get();
        $htmlView = view('admin.inventory.edit',compact('inventory','allproduct','allarea'))->render();
        return response()->json(['success' => true, 'htmlView' => $htmlView,'entry_date' => $inventory->entry_date]);
    }   

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateRequest $request, Inventory $inventory)
    {
        $oldvalue = $inventory->getOriginal(); 
        $inventory->update($request->all());
        $newValue = $inventory->getAttributes();
        addToLog($request,'Inventory','Edit', $newValue ,$oldvalue); 

        return response()->json(['success' => true,
        'message' => trans('messages.crud.update_record'),
        'alert-type'=> trans('quickadmin.alert-type.success'),
        'title' => trans('quickadmin.inventory.inventory'),
        ], 200);
    }

    public function destroy(Inventory $inventory)
    {
        abort_if(Gate::denies('inventory_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $oldProductData = Inventory::where('uuid',$inventory->uuid)->first();
        if($oldProductData){
            $IsSameProduct = Inventory::where('product_id',$oldProductData->product_id)->where('place',$oldProductData->place)->where('uuid','<>',$inventory->uuid)->sum('quantity');
            $dispach_order = OrderProduct::where('product_id', $oldProductData->product_id)->where('dispatch_place',$oldProductData->place)->sum('quantity');
            if($IsSameProduct < $dispach_order ){
                return response()->json(['success' => false,
                 'message' => 'Quantity is already delivered',
                 'alert-type'=> trans('quickadmin.alert-type.error'),
                 'title' => trans('quickadmin.inventory.inventory')
                ], 200);
            }else{
                $inventory->delete();
                return response()->json(['success' => true,
                 'message' => trans('messages.crud.delete_record'),
                 'alert-type'=> trans('quickadmin.alert-type.success'),
                 'title' => trans('quickadmin.inventory.inventory')
                ], 200);
            }               
        }     
       
    }
    
    public function inventoryView(InventoryViewDataTable $dataTable)
    {
        abort_if(Gate::denies('inventory_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $type='all';
        $brands = Brand::orderBy('name', 'asc')->get();
        $storageTypes = $this->storageTypes;
        return $dataTable->render('admin.inventory.inventory_view',compact('type','brands','storageTypes'));
    }

    public function inventoryDetail($id, $place){
        abort_if(Gate::denies('inventory_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');        
        $alldata = [];
        if(isset($_REQUEST['start_date']) && isset($_REQUEST['end_date'])){
            $from_date = $_REQUEST['start_date'];
            $to_date =   $_REQUEST['end_date'];
        }else{
            // $todayDate = Carbon::now()->format('Y-m-d'); 
            $to_date = Carbon::now()->format('Y-m-d');
            $from_date = Carbon::now()->subDays(29)->format('Y-m-d');
        }  
              
        $openingBalance = inventoryOpeningBalance($id, $place,$from_date, $to_date);
        $product = Product::findOrFail($id);

        $BuyProduct = Inventory::select('uuid','product_id', 'entry_date as transaction_date','created_at','shift_id', DB::raw("NULL as shifted_to"), DB::raw("'' as invoice_number"), DB::raw("'Buy' as type"),'quantity','place')
            ->where('place', $place)
            ->where('is_active', 1)
            ->whereBetween('entry_date', [$from_date, $to_date])
            ->where('product_id', $id);

        $SellProduct = OrderProduct::select('orders.uuid','order_products.product_id', 'orders.invoice_date as transaction_date', 'order_products.created_at',DB::raw("NULL as shift_id"), DB::raw("NULL as shifted_to"),'orders.invoice_number as invoice_number', DB::raw("'Sell' as type"), 'order_products.quantity', 'order_products.dispatch_place as place')
            ->where('dispatch_place', $place)
            ->leftJoin('orders', 'orders.id', '=', 'order_products.order_id')
            ->whereBetween('orders.invoice_date', [$from_date, $to_date])
            ->where('order_products.product_id', $id);
        
        $ShiftProduct = ShiftingGood::select(
            DB::raw("NULL as uuid"),
            'product_id',
            'entry_date as transaction_date',
            'created_at',
            'id as shift_id',
            'to as shifted_to',
            DB::raw("NULL as invoice_number"),
            DB::raw("'Shift' as type"),
            'quantity',
            DB::raw('`from` as place')
        )
        ->where('product_id', $id)
        ->where(function($q) use ($place) {
            $q->where('from', $place);
        })
        ->whereBetween('entry_date', [$from_date, $to_date]);

        // Union the two queries
        $alldata = $BuyProduct->unionAll($SellProduct)->unionAll($ShiftProduct)->orderBy('transaction_date','asc')->orderBy('created_at', 'asc')->get();
        return view('admin.inventory.inventory_view_single',compact('product','openingBalance','alldata','from_date','to_date','place'));
    }

    public function storeShiftingGoods(ShiftingGoodRequest $request)
    {
        try {
            $data = ShiftingGood::create($request->validated());
            addToLog($request, 'ShiftingGood', 'Create', $data);
            $inventroyData = $request->only('product_id', 'quantity', 'entry_date');
            $inventroyData['place'] = $request->to;
            $inventroyData['shift_id'] = $data->id;
            $inventroyData['type'] = 'shift_buy';
            Inventory::create($inventroyData);
            addToLog($request,'Inventory','Create', $inventroyData);
            return response()->json([
                'success' => true,
                'message' => trans('messages.crud.add_record'),
                'alert-type' => trans('quickadmin.alert-type.success'),
                'title' => trans('quickadmin.inventory.shifting_goods'),
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'alert-type' => trans('quickadmin.alert-type.error'),
                'title' => trans('quickadmin.inventory.shifting_goods'),
            ], 500);
        }
    }

    public function allInventoryPrintView(Request $request)
    {
        $listType = $request->get('listtype', 'all');

        switch ($listType) {
            case 'all':
                $query = Inventory::query()
                    ->select(
                        'inventories.product_id',
                        'inventories.place',
                        DB::raw('SUM(inventories.quantity) - IFNULL((
                            SELECT SUM(quantity)
                            FROM shifting_goods
                            WHERE shifting_goods.product_id = inventories.product_id
                            AND shifting_goods.from = inventories.place
                        ), 0) as total_buy')
                    )
                    ->with('product', 'product.brands')
                    ->where('inventories.is_active', 1)
                    ->groupBy('inventories.product_id', 'inventories.place')
                    ->orderBy('product_id');
                break;

            default:
                return abort(404);
        }

        // apply brand filter
        if ($request->brand_id) {
            $query->whereHas('product', function ($q) use ($request) {
                $q->where('brand_id', $request->brand_id);
            });
        }

        // apply storage filter
        if ($request->storage_type) {
            $query->where('place', $request->storage_type);
        }

        $rows = $query->get();

        // get products that match filter (even if no inventory rows exist)
        $productQuery = Product::with('brands');
        if ($request->brand_id) {
            $productQuery->where('brand_id', $request->brand_id);
        }
        $products = $productQuery->get()->keyBy('id');

        // storage types
        $places = array_keys(config('app.product_place', []));
        if ($request->filled('storage_type')) {
            $places = [$request->storage_type];
        }

        $finalRows = collect();
        
        foreach ($products as $productId => $product) {
            foreach ($places as $place) {
                $existing = $rows->firstWhere(fn($row) => $row->product_id == $productId && $row->place == $place);

                $row = new \stdClass();
                $row->product_id = $productId;
                $row->place = $place;
                $row->total_buy = $existing ? (int) $existing->total_buy : 0;
                $row->product = $product;
                $finalRows->push($row);
            }
        }

        // apply selected checkboxes if passed
        if ($request->filled('inventory_id')) {
            $selectedIds = explode(',', $request->inventory_id);
            $finalRows = $finalRows->filter(function ($row) use ($selectedIds) {
                return in_array($row->product_id . '|' . $row->place, $selectedIds);
            })->values();
        }

        $finalRows = $finalRows->sortBy(fn($row) =>
            strtolower($row->product->brands->name ?? '') . '|' . strtolower($row->product->name ?? '')
        )->values();

        $places = config('app.product_place');
        $inventories = $finalRows;

        return view('admin.inventory.print-inventory-list', compact('inventories', 'places'))->render();
    }


}
