<?php

namespace App\DataTables;

use App\Models\Inventory;
use App\Models\Product;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\Gate;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class ProductDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */

     protected $isRecycle;
     public function withParam1($recycle)
     {
         $this->isRecycle = $recycle;
         return $this;
     }

    public function dataTable(QueryBuilder $query)
    {       
        $openingBalanceSql = "(
            (
                SELECT COALESCE(SUM(i.quantity),0) 
                FROM inventories i
                WHERE i.product_id = products.id 
                AND i.is_active = 1 
            )
            -
            (
                SELECT COALESCE(SUM(op.quantity),0) 
                FROM order_products op
                LEFT JOIN orders o ON o.id = op.order_id
                WHERE op.product_id = products.id 
            )
            -
            (
                SELECT COALESCE(SUM(sg.quantity),0) 
                FROM shifting_goods sg
                WHERE sg.product_id = products.id 
            )
        )";
        return datatables()
            ->eloquent($query)
            ->addIndexColumn()
            ->editColumn('name',function($row){
                return $row->name ?? "";
            })
            ->editColumn('brands.name',function($row){
                return $row->brands->name;
            })
            ->editColumn('product_unit.name',function($row){
                return  $row->product_unit ? $row->product_unit->name : '';
            })
            ->editColumn('calculation',function($row){
                return  $row->calculation ?? '';
            })
            ->editColumn('opening_balance',function($row){
                return  (int)$row->opening_balance ?? '';
            })
            ->editColumn('created_at', function ($row) {
                return $row->created_at->format('d-m-Y h:i A');
            })
            ->addColumn('action',function($row){
                $checkAvailabily = Inventory::where('product_id', $row->id)->exists();
                $action='<div>';
                if($this->isRecycle == "isRecycle"){
                    if (Gate::check('product_undo')) {
                        $editIcon = '<i class="fa fa-undo" aria-hidden="true"></i>';
                        $action .= '<a href="javascript:void(0)" title="Edit" class="btn btn-icon btn-info recycle_group" data-id="'.encrypt($row->id).'">'.$editIcon.'</a>';
                    }
                }else{
                    if (Gate::check('product_access')) {
                        $editIcon = view('components.svg-icon', ['icon' => 'view'])->render();
                       $action .= '<a href="javascript:void(0)" title="View Detail" class="btn btn-icon btn-info view_detail" data-id="'.encrypt($row->id).'" data-product_name="'.$row->name.'" >'.$editIcon.'</a>';
                   }
                    if (Gate::check('product_edit')) {
                         $editIcon = view('components.svg-icon', ['icon' => 'edit'])->render();
                         $action .= '<button class="btn btn-icon btn-info edit-data-btn p-1" title="Edit" data-id="'.encrypt($row->id).'" data-href="'.route('products.edit',$row->id).'">'.$editIcon.'</button>';
                    }

                    if (Gate::check('product_delete')) {
                        if($row->order_count == 0 && $checkAvailabily == false){
                            $deleteIcon = view('components.svg-icon', ['icon' => 'delete'])->render();
                            $action .= '<a href="javascript:void(0)" title="Delete" class="btn btn-icon btn-danger delete_product" data-id="'.encrypt($row->id).'">  '.$deleteIcon.'</a>';
                        }
                    }
                }
                $action .='</div>';
                return $action;
            })
            ->filterColumn('created_at', function ($query, $keyword) {
                $query->whereRaw("DATE_FORMAT(products.created_at,'%d-%M-%Y') like ?", ["%$keyword%"]); //date_format when searching using date
            })
            ->filterColumn('brands.name', function ($query, $keyword) {
                $query->whereHas('brands', function ($q) use ($keyword) {
                    $q->where('brands.name', 'like', "%$keyword%");
                });
            })
            ->orderColumn('brands.name', function ($query, $order) {
                $query->orderBy('brands.name', $order);
            })                  
            ->orderColumn('product_unit.name', function ($query, $order) {
                $query->orderBy('unit_types.name', $order);
            })    
            ->filterColumn('opening_balance', function($query, $keyword) use ($openingBalanceSql) {
                $query->whereRaw("($openingBalanceSql) LIKE ?", ["%{$keyword}%"]);
            })    
            ->orderColumn('opening_balance', function($query, $order) use ($openingBalanceSql) {
                $query->orderByRaw("($openingBalanceSql) {$order}");
            })  
            ->rawColumns(['action','calculation_type','price','min_sale_price','wholesaler_price','retailer_price']);
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Product $model): QueryBuilder
    {
        $query = $model->newQuery()->select(['products.*', DB::raw("(
                (
                    SELECT COALESCE(SUM(i.quantity),0) 
                    FROM inventories i
                    WHERE i.product_id = products.id 
                      AND i.is_active = 1 
                )
                -
                (
                    SELECT COALESCE(SUM(op.quantity),0) 
                    FROM order_products op
                    LEFT JOIN orders o ON o.id = op.order_id
                    WHERE op.product_id = products.id 
                )
                -
                (
                    SELECT COALESCE(SUM(sg.quantity),0) 
                    FROM shifting_goods sg
                    WHERE sg.product_id = products.id 
                )
            ) as opening_balance")])->with(['brands', 'product_unit']);

        if($this->isRecycle == "isRecycle"){
            $query->onlyTrashed();
        }

        return $this->applyScopes($query);
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
                    ->setTableId('products-table')
                    ->parameters([
                        'responsive' => false,
                        'pageLength' => 50,
                    ])
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    ->dom('frtip')
                    ->orderBy(6);
                    // ->selectStyleSingle()

    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [

            Column::make('DT_RowIndex')->title(trans('quickadmin.qa_sn'))->orderable(false)->searchable(false),
            Column::make('name')->title(trans('quickadmin.product2.fields.name'))->orderable(true),
            Column::make('brands.name')->title(trans('quickadmin.product2.fields.brand')),
            Column::make('product_unit.name')->title(trans('quickadmin.product2.fields.unit_type')),
            Column::make('calculation')->title(trans('quickadmin.product2.fields.calculation')), 
            Column::make('opening_balance')->title(trans('quickadmin.product2.fields.total_stock')), 
            Column::make('created_at')->title(trans('quickadmin.order.fields.created_at')),          
            Column::computed('action')
            ->exportable(false)
            ->printable(false)
            ->width(60)
            ->addClass('text-center')->title(trans('quickadmin.qa_action')),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'product' . date('YmdHis');
    }
}
