<?php

namespace App\DataTables;

use App\Models\OrderEditHistory;
use App\Models\PaymentTransaction;
use App\Models\PaymentTransactionHistory;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\Gate;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class PaymentTransactionDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */

    public function dataTable(QueryBuilder $query)
    {
        return datatables()
            ->eloquent($query)            
            ->addIndexColumn()
            ->editColumn('entry_date', function ($row) {
                return date('d-m-Y', strtotime($row->entry_date)) ?? "";
            })           
            ->editColumn('customer.name',function($row){
                return $row->customer->name;
            })
            ->editColumn('voucher_number', function ($row) {
                return $row->voucher_number ?? "";
            })
            ->editColumn('payment_way', function ($row) {
                return (isset(config('constant.paymentModifyWays')[$row->payment_way]) ? config('constant.paymentModifyWays')[$row->payment_way] : '');
            })            
            ->editColumn('credit_amount', function ($row) {
                return ($row->payment_type == 'credit' ? number_format(abs($row->amount), 2) : '');
            })
            ->editColumn('created_at', function ($row) {
                return $row->created_at ? $row->created_at->format('d-m-Y h:i A') : '';
            })
            ->addColumn('action', function ($row) {
                $action = '<div>';            
                
                if (Gate::check('transaction_access')) {
                    $viewIcon = view('components.svg-icon', ['icon' => 'view'])->render();
                    $action .= '<button data-href="' . route('transactions.show', $row->uuid) . '"  class="btn btn-icon btn-info view_cashreceipt" title="'.trans('quickadmin.qa_view').'" >' . $viewIcon . '</button>';                        
                }  
              
                if (Gate::check('transaction_edit')) {
                    $editIcon = view('components.svg-icon', ['icon' => 'edit'])->render();
                    $action .= '<a href="' . route("transactions.edit", $row->uuid) . '" class="btn btn-icon btn-info edit_product" title="'.trans('quickadmin.qa_update').'" >' . $editIcon . '</a>';
                }
                
                if (Gate::check('transaction_delete')) {                
                    $deleteIcon = view('components.svg-icon', ['icon' => 'delete'])->render();
                    $action .= '<button title="'.trans('quickadmin.qa_delete').'" class="btn btn-icon btn-info delete_transaction p-1" data-href="'.route('transactions.destroy', $row->uuid).'">'.$deleteIcon.'</button>';                
                }
                $action .= '</div>'; 
                return $action;
            })
            ->filterColumn('customer_name', function ($query, $keyword) {
                $query->whereHas('customer', function ($q) use ($keyword) {
                    $q->where('customers.name', 'like', "%$keyword%");
                });
            })           
            ->orderColumn('customer.name', function ($query, $order) {
                $query->orderBy('customers.name', $order);
            }) 

            // ->filterColumn('payment_way', function ($query) {
            //     $searchValue = isset(request()->search['value']) ? request()->search['value'] : null;
            //     $paymentKey = array_search(ucwords(trim($searchValue)), array_map('ucwords', config('constant.paymentModifyWays')));
            //     if ($paymentKey !== false) {
            //         $query->where('payment_way', $paymentKey); 
            //     }
            // }) 
           
            ->filterColumn('created_at', function ($query, $keyword) {
                $query->whereRaw("DATE_FORMAT(payment_transactions.created_at,'%d-%M-%Y') like ?", ["%$keyword%"]); //date_format when searching using date
            }) 
            ->setRowClass(function ($row) {                
                return $row->is_modified ? 'estimates-trans-active' : '';                
            })
            ->rawColumns(['action','checkbox']);
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(PaymentTransaction $model): QueryBuilder
    {     
        $query = $model->newQuery()
                   ->select('payment_transactions.*') // Select all from the current model
                   ->with(['customer' => function($query) {
                       $query->select('id', 'name', 'area_id')->whereNull('deleted_at'); // Select specific fields from customer
                   }]);
                //    ->whereIn('payment_way', ['by_cash', 'by_check', 'by_account','by_phonepay']);
        return $this->applyScopes($query);
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('payment_transaction-table')
            ->parameters([
                'responsive' => false,
                'pageLength' => 50,
                'orderClasses' => false,
            ])
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->dom('frtip')
            ->orderBy(1);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
       
       return [
            Column::make('DT_RowIndex')->title(trans('quickadmin.qa_sn'))->titleAttr(trans('quickadmin.qa_sn'))->orderable(false)->searchable(false),
            Column::make('id')->visible(false),
            Column::make('entry_date')->title(trans('quickadmin.order.fields.estimate_date'))->titleAttr(trans('quickadmin.order.fields.estimate_date'))->searchable(false),
            Column::make('customer.name')->title(trans('quickadmin.transaction.fields.customer'))->titleAttr(trans('quickadmin.transaction.fields.customer'))->orderable(true),
            Column::make('voucher_number')->title(trans('quickadmin.estimate_number'))->titleAttr(trans('quickadmin.estimate_number')),
            // Column::make('payment_way')->title(trans('quickadmin.transaction.fields.payment_type'))->titleAttr(trans('quickadmin.transaction.fields.payment_type')),
            Column::make('credit_amount')->title(trans('quickadmin.transaction.fields.credit_amount'))->name('amount')->titleAttr(trans('quickadmin.transaction.fields.credit_amount')),
            Column::make('created_at')->title(trans('quickadmin.transaction.fields.created_at'))->titleAttr(trans('quickadmin.transaction.fields.created_at'))->searchable(true),
            Column::computed('action')
                        ->exportable(false)
                        ->printable(false)
                        ->width(60)
                        ->addClass('text-center')->title(trans('quickadmin.qa_action'))
        ];       
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'order' . date('YmdHis');
    }
}
