<?php

namespace App\DataTables;

use App\Models\Product;
use App\Models\Inventory;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;

class InventoryViewDataTable extends DataTable
{
    public $paramValue = null;    
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query)
    {
        $rows = $query->get();
        $productIds = $rows->pluck('product_id')->unique()->toArray();
        $products = Product::with('brands')->whereIn('id', $productIds)->get()->keyBy('id');
        $places = array_keys(config('app.product_place', []));
        $finalRows = collect();
        foreach ($products as $productId => $product) {
            foreach ($places as $place) {
                $existing = $rows->firstWhere(fn($row) => $row->product_id == $productId && $row->place == $place);

                $row = new \stdClass();
                $row->product_id = $productId;
                $row->place = $place;
                $row->total_buy = $existing ? $existing->total_buy : 0;
                $row->product = $product;

                $finalRows->push($row);
            }
        }
        $finalRows = $finalRows->sortBy(function ($row) {
            $brandName = strtolower($row->product->brands->name ?? '');
            $productName = strtolower($row->product->name ?? '');
            return $brandName . '|' . $productName;
        })->values();

        $dataTable = datatables()
            ->collection($finalRows)
            ->addIndexColumn()
            ->addColumn('checkbox', function ($row) {
                $value = $row->product_id . '|' . $row->place;
                return '<input type="checkbox" class="dt_checkbox" name="inventory_ids[]" value="' . $value . '">';
            })

            ->editColumn('product.name', function ($row) {
                return $row->product->name ?? '';
            })
            ->editColumn('place', function ($row) {
                return config('app.product_place.' . $row->place);
            })
            ->editColumn('product.brands.name', function ($row) {
                return $row->product->brands->name ?? '';
            })
            ->editColumn('total_buy', function ($row) {
                return ($row->total_buy > 0) ? (int)$row->total_buy : 0;
            })
            ->editColumn('total_sell', function ($row) {
                if (request()->filled('from_date') && request()->filled('to_date')) {
                    $total_sell = getSellProductByPlace($row->product_id, $row->place, request()->from_date, request()->to_date);
                } else {
                    $total_sell = getSellProductByPlace($row->product_id, $row->place);
                }              
                return ($total_sell > 0) ? (int)$total_sell : 0;
            })
            ->editColumn('total', function ($row) {
                if (request()->filled('from_date') && request()->filled('to_date')) {
                    $total_sell = getSellProductByPlace($row->product_id, $row->place, request()->from_date, request()->to_date);
                } else {
                    $total_sell = getSellProductByPlace($row->product_id, $row->place);
                } 
                return ((int)$row->total_buy - (int)$total_sell) ?? 0;
            })
            ->addColumn('action', function($row) {
                $action = '';
                if (Gate::check('inventory_access')) {
                    $editIcon = view('components.svg-icon', ['icon' => 'view'])->render();
                    $action .= '<a href="'.route('inventory_detail', ['id'=>$row->product_id, 'place'=>$row->place]).'" title="View Detail" class="btn btn-icon btn-info">'.$editIcon.'</a>';
                }
                return $action;
            });

            $dataTable->filter(function ($instance) use ($finalRows) {
                $brandId = request()->get('brand_id');
                $storageType = request()->get('storage_type');
                $searchValue = request()->get('search')['value'] ?? '';
                $rows = $finalRows->map(function ($row) {
                $total_sell = request()->filled('from_date') && request()->filled('to_date')
                    ? getSellProductByPlace($row->product_id, $row->place, request()->from_date, request()->to_date)
                    : getSellProductByPlace($row->product_id, $row->place);

                $checkboxValue = $row->product_id . '|' . $row->place;
                $checkbox = '<input type="checkbox" class="dt_checkbox" name="inventory_ids[]" value="' . $checkboxValue . '">';

                $action = '';
                if (Gate::check('inventory_access')) {
                    $editIcon = view('components.svg-icon', ['icon' => 'view'])->render();
                    $action .= '<a href="'.route('inventory_detail', ['id'=>$row->product_id, 'place'=>$row->place]).'" title="View Detail" class="btn btn-icon btn-info">'.$editIcon.'</a>';
                }

                return [
                    'id'         => $checkboxValue,
                    'checkbox'   => $checkbox, 
                    'product_id' => $row->product_id,
                    'place'      => config('app.product_place.' . $row->place, $row->place),
                    'total_buy'  => (int) $row->total_buy,
                    'total_sell' => (int) $total_sell,
                    'total'      => (int) $row->total_buy - (int) $total_sell,
                    'product'    => $row->product ? $row->product->toArray() : [],
                    'action'     => $action,
                    'raw_place'  => $row->place,
                    'brand_id'   => $row->product->brand_id ?? null,
                ];
            });


                $instance->collection = $rows->filter(function ($row) use ($brandId, $storageType, $searchValue) {
                    $matches = true;
                    if (!empty($brandId)) {
                        $matches = $matches && ($row['brand_id'] == $brandId);
                    }
                    if (!empty($storageType)) {
                        $matches = $matches && ($row['raw_place'] == $storageType);
                    }

                    if (!empty($searchValue)) {
                        $matches = $matches && app(InventoryViewDataTable::class)->filterRow($row, $searchValue);
                    }

                    return $matches;
                })->values();
            }, true);



        return $dataTable->rawColumns(['action', 'invoice_number', 'product.name','checkbox']);
    }

    /**
     * Check if value is numeric including negative numbers
     */
    protected function isNumericWithNegative($value)
    {
        // Allow negative numbers with - symbol
        if (preg_match('/^-?\d+$/', $value)) {
            return true;
        }
        return false;
    }

    /**
     * Filter individual row based on search value
     */
    protected function filterRow($row, $searchValue)
    {
        $searchLower = strtolower($searchValue);
        $matches = [];
        
        if (stripos($row['place'], $searchValue) !== false) {
            $matches[] = true;
        }
        
        $brandName = strtolower($row['product']['brands']['name'] ?? '');
        if (stripos($brandName, $searchLower) !== false) {
            $matches[] = true;
        }
        
        $productName = strtolower($row['product']['name'] ?? '');
        if (stripos($productName, $searchLower) !== false) {
            $matches[] = true;
        }

        if ($this->isNumericWithNegative($searchValue)) {
            $searchNumber = (int)$searchValue;

            $totalBuy = isset($row['total_buy']) ? (int)$row['total_buy'] : null;
            $totalSell = isset($row['total_sell']) ? (int)$row['total_sell'] : null;
            $total = isset($row['total']) ? (int)$row['total'] : null;

            if ($totalBuy === $searchNumber || $totalSell === $searchNumber || $total === $searchNumber) {
                $matches[] = true;
            }
        } else {
            if (
                $row['total_buy'] == $searchValue ||
                $row['total_sell'] == $searchValue ||
                $row['total'] == $searchValue
            ) {
                $matches[] = true;
            }
        }
        // if($row['total_buy'] == 50) dd($searchValue, $row, $row['total_buy'] == $searchValue);
        return !empty($matches);
    }
    /**
     * Get the query source of dataTable.
     */
    public function query(Inventory $model): QueryBuilder
    {
        $query = $model->newQuery()
            ->select(
                'inventories.product_id',
                'inventories.place',
                DB::raw('
                    SUM(inventories.quantity) - IFNULL((
                        SELECT SUM(quantity)
                        FROM shifting_goods
                        WHERE shifting_goods.product_id = inventories.product_id
                        AND shifting_goods.from = inventories.place
                    ), 0) as total_buy
                ')
            )
            ->leftJoin('products', 'products.id', '=', 'inventories.product_id')
            ->leftJoin('brands', 'brands.id', '=', 'products.brand_id')
            ->where('inventories.is_active', 1)
            ->with('product', 'product.brands')
            ->groupBy('inventories.product_id', 'inventories.place')
            ->orderBy('brands.name', 'asc')
            ->orderBy('products.name', 'asc');

        if (request()->filled('from_date') && request()->filled('to_date')) {
            $query->whereBetween('inventories.entry_date', [request()->from_date, request()->to_date]);
        }

        return $query;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('orders-table')
            ->parameters([
                'responsive' => false,
                'pageLength' => 50,
                // 'serverSide' => false, // Set to false for collection filtering
                'ordering' => false,
            ])
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->dom('frtip');
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns()
    {
        return [
            Column::computed('checkbox')->title('<label class="custom-checkbox"><input type="checkbox" id="dt_cb_all" ><span></span></label>')->titleAttr('')->orderable(false)->searchable(false),
            Column::make('DT_RowIndex')->title(trans('quickadmin.qa_sn'))->orderable(false)->searchable(false)->visible(false),
            Column::make('place')->title(trans('quickadmin.inventory.fields.storage_type'))->orderable(false),
            Column::make('product.brands.name')->title(trans('quickadmin.inventory.fields.brand_name')),
            Column::make('product.name')->title(trans('quickadmin.inventory.fields.product_name')),
            Column::make('total_buy')->title(trans('quickadmin.inventory.fields.total_buy'))->orderable(false),
            Column::make('total_sell')->title(trans('quickadmin.inventory.fields.total_sell'))->orderable(false),
            Column::make('total')->title(trans('quickadmin.inventory.fields.total_left'))->orderable(false),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center')->title(trans('quickadmin.qa_action')),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Invoice_' . date('YmdHis');
    }
}
